% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rolr-functions.R
\name{rlr}
\alias{rlr}
\title{Calculating Running Logrank Test}
\usage{
rlr(times, status, x, ns = 15, trend = "decrease", method = "approximate")
}
\arguments{
\item{times}{Survival outcome.}

\item{status}{Censoring indicator which takes 1 when an event occurs at the
end of a study and 0 otherwise.}

\item{x}{A continuous covariate.}

\item{ns}{Minimum number of subjects in each group, whether it is the group
with \code{x < cutpoint}, or the group with \code{x >= cutpoint}.}

\item{trend}{A character that takes either \code{"decrease"} or \code{"increase"}
to represent a positive or negative relationship between the covariate
and survival.}

\item{method}{A character that takes either \code{"approximate"} or \code{"exact"}
where an approximate or exact method will be used to calculate the running
logrank test.}
}
\value{
A matrix of four columns as the following -

xcutoff - All cutpoints that have been used to dichotomize the sample
  (that is, all values of the covariate x except the first and last \code{ns} points)

L - Numerators of the logrank z tests for all cutpoints considered.

V - Denominators of the logrank z tests for all cutpoints considered.

logrank.stat - The logrank z tests for all cutpoints considered.
}
\description{
\code{rlr} is used to calculate a logrank test for every two groups
  obtained from dichotomizing a continuous covariate x at a particular point.
  It will examine all values in x except the first and last \code{ns} points.
}
\details{
When the association is positive, that is, larger covariate values
  leading to worse survival, and you enter \code{trend = "decrease"}, the test
  statistics will be positive, but if you enter \code{trend = "increase"} the
  test statistics will be negative. Opposite is true when the association
  is negative. You want to make sure to enter the option so that the
  resulting test statistics are positive.
}
\examples{
library(rolr)

##### -------- Example 1

#simulate survival where hazard increases as covariate increases
d=simdata(nn = 150, hr.linear = 2, hazard.func = "linear", censoring.rate = 0)

#using trend = 'decrease', the test statistics are positive, which is good
res=rlr(times=d$times, status=d$status, x=d$x, ns=15, trend='decrease')
head(res)

#do it again with trend = 'increase', now the test statistics are negative.
#So you want to switch to trend='decrease'.
res2=rlr(times=d$times, status=d$status, x=d$x, ns=15, trend='increase')
head(res2)

#Note that the test statistics are the same except the signs
res[,'logrank.stat']+res2[,'logrank.stat']

#do it with exact method, how close is it to the approximate method?
res3=rlr(times=d$times, status=d$status, x=d$x, ns=15, trend='decrease',
         method="exact")
cor(res[,'logrank.stat'], res3[,'logrank.stat'])

##### -------- Example 2

#Simulate survival where hazard decreases as covariate increases
d=simdata(nn = 150, hr.linear = 1/3, hazard.func = "linear", censoring.rate = 0)

#using trend = 'decrease', and the test statistics are negative, which
#is not good
res=rlr(times=d$times, status=d$status, x=d$x, ns=15, trend='decrease')
head(res)

#do it again with trend = 'increase', now the test statistics are positive,
#which is good
res2=rlr(times=d$times, status=d$status, x=d$x, ns=15, trend='increase')
head(res2)

#Note that the test statistics are the same except the signs
res[,'logrank.stat']+res2[,'logrank.stat']

#do it with exact method, how close is it to the approximate method?
res3=rlr(times=d$times, status=d$status, x=d$x, ns=15, trend='increase',
         method="exact")
cor(res[,'logrank.stat'], res3[,'logrank.stat'])

}
\references{
See main package help page.
}

