% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extract.R
\name{extract}
\alias{extract}
\title{Create geographical extracts from an OSM file}
\usage{
extract(
  input_path,
  extent,
  output_path,
  strategy = c("complete_ways", "smart", "simple"),
  overwrite = FALSE,
  echo_cmd = FALSE,
  echo = TRUE,
  spinner = TRUE,
  verbose = FALSE
)
}
\arguments{
\item{input_path}{A string. The path to the OSM data/history file whose
extent should be extracted from. Please see \link{file_formats} for a list of
supported file formats.}

\item{extent}{Either a \code{POLYGON} or a \code{MULTIPOLYGON} \code{sf} object with only
one feature or a \code{bbox} object, created by \link[sf:st_bbox]{sf::st_bbox}.}

\item{output_path}{A string. The path to the file where the output should be
written to. Please see \link{file_formats} for a list of supported file formats.}

\item{strategy}{A string. The strategy to use when creating the extract.
Available strategies are \code{"complete_ways"}, \code{"smart"} and \code{"simple"}.
Defaults to \code{"complete_ways"}. Please see the "Strategies" section for a
description of each one of them.}

\item{overwrite}{A logical. Whether existing files should be overwritten by
the output. Defaults to \code{FALSE}.}

\item{echo_cmd}{A logical. Whether to print the Osmium command generated by
the function call to the screen. Defaults to \code{FALSE}.}

\item{echo}{A logical. Whether to print the standard output and error
generated by the Osmium call to the screen. Defaults to \code{TRUE}.}

\item{spinner}{A logical. Whether to show a reassuring spinner while the
Osmium call is being executed. Defaults to \code{TRUE}.}

\item{verbose}{A logical. Whether to display detailed information on the
running command. Defaults to \code{FALSE}.}
}
\value{
The normalized path to the output file.
}
\description{
Creates geographical extracts from an OSM data file or an OSM history file.
The geographical extent can be given either as a bounding box or as a
(multi)polygon.
}
\section{Strategies}{

Different strategies can be used when creating extracts. Depending on the
strategy, different objects will end up in the extracts. The strategies
differ in how much memory they need and how often they need to read the input
file. The choice of strategy depends on how you want to use the generated
extracts and how much memory and time you have.
\itemize{
\item \code{"simple"} - runs in a single pass. The extract will contain all nodes
inside the region and all ways referencing those nodes, as well as all
relations referencing any nodes or ways already included. Ways crossing the
region boundary will not be reference-complete. Relations will not be
reference-complete. This strategy is fast, because it reads the input only
once, but the result is not enough for most use cases. This strategy will not
work for history files.
\item \code{"complete_ways"} - runs in two passes. The extract will contain all nodes
inside the region and all ways referencing those nodes, as well as all nodes
referenced by those ways. The extract will also contain all relations
referenced by nodes inside the region or ways already included and,
recursively, their parent relations. The ways are reference-complete, but the
relations are not.
\item \code{"smart"} - runs in three passes. The extract will contain all nodes inside
the region and all ways referencing those nodes, as well as all nodes
referenced by those ways. The extract will also contain all relations
referenced by nodes inside the region or ways already included and,
recursively, their parent relations. The extract will also contain all nodes
and ways (and the nodes they reference) referenced by relations tagged
"type=multipolygon" directly referencing any nodes in the region or ways
referencing nodes in the region. The ways are reference-complete, and all
multipolygon relations referencing nodes in the regions or ways that have
nodes in the region are reference-complete. Other relations are not
reference-complete.
}
}

\examples{
\dontshow{if (identical(tolower(Sys.getenv("NOT_CRAN")), "true")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
pbf_path <- system.file("extdata/cur.osm.pbf", package = "rosmium")

file.size(pbf_path)

# buffering the pbf bounding box 4000 meters inward and using the result
# extent to extract the osm data inside it. transforming the crs because
# inward buffers only work with projected crs

lines <- sf::st_read(pbf_path, layer = "lines", quiet = TRUE)
bbox <- sf::st_bbox(lines)
bbox_polygon <- sf::st_as_sf(sf::st_as_sfc(bbox))
smaller_bbox_poly <- sf::st_buffer(
  sf::st_transform(bbox_polygon, 5880),
  -4000
)
smaller_bbox_poly <- sf::st_transform(smaller_bbox_poly, 4326)

output_path <- extract(
  pbf_path,
  smaller_bbox_poly,
  tempfile(fileext = ".osm.pbf")
)

file.size(output_path)
\dontshow{\}) # examplesIf}
}
