#' Graduates raw death probabilities
#'
#' @description This function graduates raw death probabilities using the two-step
#'              approach proposed by Sifre-Armengol et al. (2024). The method
#'              involves (i) applying natural cubic splines to smooth the
#'              log-transformation of the raw
#'              estimates, and (ii) correcting unexpected decreases in mortality
#'              trends at the oldest ages by modelling and extending death
#'              probability estimates up to a higher maximum age (omega). The function accepts
#'              as main input either an output from  `dw_crude_mx` or `tw_crude_mx`, or
#'              a list with containing similar `qx`, `Lx` and `dx` components.
#'
#' @author Jose M. Pavia \email{pavia@@uv.es}
#' @references Sifre-Armengol, C., Pavia J.M. and Lledo, J. (2025) A comprehensive database of estimates and forecasts of Spanish sex-age death rates by climate area, income level, and habitat size (2010-2050). *Demographic Research*, 52(1): 1-24. \doi{10.4054/DemRes.2025.52.1}.
#'
#' @param x A list that is either an output of the `dw_crude_mx` or `tw_crude_mx` functions, or a list containing similar `qx`, `Lx` and `dx` components as the outputs of these functions. It is assumed that the first dimension of the matrices/arrays in `x` corresponds to age. Likewise, as happens with `dw_crude_mx` or `tw_crude_mx` is assumed that ages are ordered sequentially.
#' @param ages A vector of two components of non-negative integers indicating the minimum and maximum ages of the matrices in `x`.
#' @param df A non-negative integer indicating the number of degrees of freedom to be used in the natural splines. Default, the greater between 5 and the range of `ages` divided by 5. It is recommended to test several values for `df` if the smooth does not look appropriate.
#' @param threshold A non-negative integer indicating the minimum age from which start to explore whether unexpected decreases in mortality trends are observed. Default, 75.
#' @param size A non-negative integer indicating the sample size to be used for modelling (default is 10). No modelling is performed for a series of estimates if, after determining the effective threshold, the number of available observations is insufficient.
#' @param omega A non-negative integer indicating the initial maximum age up to which death probabilities are to be extended. By default, this is set to the maximum between `omega` (default value 120) and the maximum age in `x`.
#' @param zero2NA A nonnegative number to be used as a threshold for excluding particularly small raw estimated death probabilities. Default, `0`.
#' @param max.c.NA A nonnegative number informing about the maximum consequetive NA allowed in the range selected for smoothing. Default, `3`.
#' @param plot.explore A TRUE/FALSE argument indicating whether exploratory figures with the outputs should be plotted as a side-effect. Default, `FALSE`.
#' @param ... Additional parameters to be passed to the `ns` function, from the `spline` package, as this is function is used to smooth the raw probabilities.
#'
#' @return
#' A list with three components (qx.raw, qx.smooth, qx.final) with the same dimensions than qx. The first component (qx.raw) contains the introduced raw death probabilities. The second component (qx.smooth) contains the smoothed death probabilities obtained after applying the natural splines. The third (qx.final) component contains the final smoothed death probabilities attained after modelling, when possible and required.
#' @export
#'
#' @importFrom stats glm lag predict
#' @importFrom splines ns
#' @importFrom graphics plot par lines
#'
#' @note
#' When a set of final smoothed estimates cannot be obtained due to unsuccessful modeling, all corresponding values within the output component qx.final are set to `NA`.
#'
#' @examples
#'
#' qx <- matrix(c(0.00053, 1e-04, 0.00013, 1e-04, 9e-05, 6e-05, 4e-05, 3e-05,
#'                4e-05, 9e-05, 8e-05, 0.00017, 0.00025, 0.00031, 0.00029,
#'                0.00034, 0.00035, 0.00035, 0.00036, 0.00034, 0.00035, 0.00034,
#'                0.00037, 0.00037, 0.00045, 0.00048, 5e-04, 0.00056, 0.00068,
#'                1e-04, 0.00073, 0.00077, 0.00073, 0.00074, 0.00081, 0.00086,
#'                0.00099, 0.00107, 0.00111, 0.00116, 0.00122, 0.00142, 0.00167,
#'                0.00197, 0.00235, 0.00282, 0.00305, 0.00327, 0.00357, 0.0039,
#'                0.00438, 0.00481, 0.00571, 0.00615, 0.00676, 0.00728, 0.00787,
#'                0.00839, 0.0094, 0.01044, 0.01119, 0.01231, 0.01336, 0.01456,
#'                0.01551, 0.0168, 0.01834, 0.02012, 0.0214, 0.02341, 0.02565,
#'                0.02744, 0.02995, 0.03275, 0.03604, 0.04027, 0.04617, 0.05092,
#'                0.05793, 0.06425, 0.07136, 0.07907, 0.0878, 0.09703, 0.10735,
#'                0.11913, 0.13115, 0.14478, 0.15927, 0.17506, 0.19133, 0.20803,
#'                0.22459, 0.23739, 0.25289, 0.26508, 0.26514, 0.27186, 0.28433,
#'                0.27071, 0.23445, 0.20622, 0.17334, 0.13063, 0.12384, 0.1298,
#'                0.14223, 0.10746, 0.1071, 0.09006, 0.08212, 0.00044, 0.00015,
#'                0.00011, 0.00014, 0.00011, 6e-05, 3e-05, 4e-05, 5e-05, 0.00011,
#'                1e-04, 0.00016, 0.00021, 0.00027, 0.00028, 0.00029, 0.00035,
#'                0.00036, 0.00037, 0.00034, 0.00038, 0.00036, 0.00038, 0.00037,
#'                0.00044, 5e-04, 0.00051, 0.00055, 7e-04, 0.00074, 0.00076, 0.00078,
#'                0.00076, 7e-04, 0.00078, 0.00078, 0.00091, 0.00096, 0.00103,
#'                0.00112, 0.0012, 0.00138, 0.0016, 0.00192, 0.00224, 0.00266,
#'                0.00291, 0.00319, 0.00349, 0.00387, 0.0044, 0.00491, 0.00565,
#'                0.00609, 0.00673, 0.00723, 0.00778, 0.00844, 0.00952, 0.01044,
#'                0.01119, 0.0124, 0.01357, 0.01488, 0.01594, 0.01727, 0.01872,
#'                0.02029, 0.02171, 0.0237, 0.02625, 0.02818, 0.03051, 0.03323,
#'                0.03698, 0.04079, 0.04605, 0.05167, 0.05898, 0.06439, 0.07144,
#'                0.07935, 0.08765, 0.09585, 0.10583, 0.11705, 0.12861, 0.14218,
#'                0.15571, 0.17194, 0.18712, 0.20318, 0.21773, 0.2308, 0.24627,
#'                0.25896, 0.26325, 0.26827, 0.2765, 0.27272, 0.24799, 0.22936,
#'                0.19868, 0.17734, 0.15662, 0.15714, 0.14942, 0.12339, 0.14107,
#'                0.08829, 0.06772),
#'                nrow = 111L, ncol = 2L, dimnames = list(NULL, c("Y2021", "Y2022")))
#'
#' Lx <- matrix(c(39493.05, 48856.36, 55123.06, 60751.17, 66031.82, 70946.31,
#'                75208.69, 79412.74, 82652.24, 85128.87, 87020.23, 88685.67,
#'                89705, 90517.94, 91646.01, 92918.48, 94170.36, 95460.83, 97687.18,
#'                100164.66, 103261.96, 106992.27, 111921.92, 116603.89, 121291.46,
#'                125936.06, 130752.09, 134804.76, 139320.39, 144206.5, 149682.01,
#'                154873.73, 160815.58, 167089.22, 173680.55, 179504.54, 184932.72,
#'                189330.61, 192820.72, 193978.13, 193934.13, 193179.32, 191721.7,
#'                188989.66, 186417.68, 184378.77, 182201.46, 180054.42, 178485.35,
#'                177261.95, 175410.36, 173603.62, 171631.32, 169582.45, 167628.59,
#'                165317.96, 162698.1, 159899.55, 156710.46, 152477.14, 148809.7,
#'                144888.18, 141481.5, 138641.49, 136507.83, 133596.61, 131455.18,
#'                129085.66, 125392.97, 120104.96, 114700.89, 107965.81, 100031.66,
#'                91822.18, 84574.49, 78886.03, 74100.05, 70091.95, 67604.06, 65699.18,
#'                62590.7, 58770.4, 54806.82, 49668.38, 43748.81, 37750.22, 31925.7,
#'                26140.64, 21029.01, 16690.26, 12842.08, 9656.27, 7150.16, 5130.6,
#'                3537.15, 2395.44, 1599.79, 1058.39, 675.82, 447.16, 308.76, 234.85,
#'                179.15, 157.42, 136.35, 122.47, 97.96, 88.06, 79.53, 63.62, 46.71,
#'                36630.78, 45891.8, 52172.94, 57352.35, 62562.67, 67386.39, 72005.75,
#'                76355.92, 80515.9, 83898.66, 86538.01, 88545.23, 90128.12, 91180.86,
#'                92148.96, 93368.73, 94695.83, 95926.17, 97595.46, 99320.7, 101814.36,
#'                104643.36, 108574.15, 112812.37, 117584.14, 121903.48, 126651.53,
#'                130739.43, 134924.59, 139100.46, 144051.46, 148808.85, 154075.99,
#'                159886.92, 166350.24, 172505.02, 178361.41, 183831.91, 188630.54,
#'                191534.35, 193004.78, 193429.58, 193222.93, 191243.25, 188992.12,
#'                186776.36, 184924.44, 182272.74, 180435.57, 179322.72, 177992.62,
#'                176083.51, 174604.92, 172917.28, 170794.39, 168626.14, 166345.52,
#'                163790.65, 160928.87, 157002.29, 152973.22, 149050.82, 144971.78,
#'                141000.87, 138484.3, 135568.46, 132464.84, 129908.86, 127041.89,
#'                122810.99, 117770.62, 111814.78, 105165.19, 97433.74, 89018.38,
#'                81824.61, 76121.28, 70589.37, 66375.32, 63833.38, 61269.47, 57992.61,
#'                54404.69, 50190.06, 45060.81, 39460.55, 33700.22, 27923.89, 22693.17,
#'                17989.68, 13907.67, 10479.36, 7768.35, 5603.56, 3927.38, 2645.58,
#'                1771.36, 1171.63, 751.12, 468.68, 307.32, 227.74, 167.73, 138.75,
#'                123.58, 105.55, 92.89, 68.44, 59.3, 54.13, 42.8),
#'                nrow = 111L, ncol = 2L, dimnames = list(NULL, c("Y2021", "Y2022")))
#'
#' dx <- matrix(c(21, 5, 7, 6, 6, 4, 3, 2, 3, 8, 7, 15, 22, 28, 27, 32, 33, 33,
#'                35, 34, 36, 36, 41, 43, 55, 61, 66, 75, 95, 101, 110, 119,
#'                118, 124, 141, 154, 183, 203, 215, 225, 237, 275, 321, 373,
#'                438, 520, 557, 589, 639, 693, 770, 837, 982, 1047, 1137, 1208,
#'                1285, 1347, 1480, 1600, 1674, 1794, 1903, 2034, 2134, 2263,
#'                2433, 2623, 2712, 2845, 2980, 3004, 3041, 3057, 3104, 3242, 3502,
#'                3662, 4033, 4361, 4632, 4838, 5033, 5065, 4963, 4782, 4481, 4080,
#'                3639, 3202, 2717, 2242, 1809, 1382, 1024, 732, 489, 333, 224,
#'                140, 82, 54, 34, 22, 18, 17, 15, 10, 9, 6, 4, 16, 7, 6, 8, 7,
#'                4, 2, 3, 4, 9, 9, 14, 19, 25, 26, 27, 33, 35, 36, 34, 39, 38,
#'                41, 42, 52, 61, 64, 72, 95, 103, 110, 116, 117, 112, 129, 134,
#'                162, 177, 194, 214, 231, 268, 310, 367, 424, 498, 539, 583, 631,
#'                696, 785, 867, 990, 1056, 1153, 1224, 1300, 1389, 1539, 1647,
#'                1722, 1860, 1981, 2114, 2225, 2361, 2503, 2663, 2788, 2946, 3132,
#'                3196, 3258, 3292, 3354, 3407, 3588, 3744, 4034, 4247, 4539, 4792,
#'                4987, 5053, 5035, 4906, 4632, 4274, 3832, 3384, 2871, 2370, 1898,
#'                1462, 1103, 787, 537, 363, 241, 148, 87, 59, 37, 27, 21, 18,
#'                15, 9, 9, 5, 3),
#'                nrow = 111L, ncol = 2L, dimnames = list(NULL, c("Y2021", "Y2022")))
#'
#' x <- list("qx" = qx, "Lx" = Lx, "dx" = dx)
#'
#' example <- graduate_qx(x = x, ages = c(3, 113))
#'

graduate_qx <- function(x,
                        ages,
                        df = max(5, round(range(ages)/5)),
                        threshold = 75,
                        size = 10,
                        omega = 120,
                        zero2NA = 0,
                        max.c.NA = 3,
                        plot.explore = FALSE,
                        ...){

  if (!("list" %in% class(x)))
    stop("The argument 'x' is not of the proper class.")

  if (sum(names(x) %in% c("qx", "Lx", "dx")) != 3)
    stop("The argument 'x' does not have the proper components.")

  if (sum(is.nan(x$qx) + is.infinite(x$qx)) > 0){
    x$qx[is.nan(x$qx) | is.infinite(x$qx)] <- NA
    warning("NaN and Infinite raw estimated death probabilities have been replaced by NA.")
  }

  if (sum(x$qx >= 1 & !is.na(x$qx)) > 0){
    x$qx[x$qx >= 1 & !is.na(x$qx)] <- 0.9999
    warning("Raw estimated death probabilities higher than 1 have been replaced by 0.9999.")
  }

  # Test inputs
  args <- c(as.list(environment()), list(...))
  test_grad(args)

  # Preliminaries
  omega <- max(omega, ages[2L])
  ages <- ages[1L]:omega
  tamanyo <- dim(x$qx)
  nombres <- dimnames(x$qx)
  nombres[[1L]] <- ages
  if (length(tamanyo) == 2L){
    qx <- qx.smooth <- qx.final <- matrix(NA, ncol = ncol(x$qx), nrow = length(ages))
    qx[1L:nrow(x$qx), ] <- x$qx[1L:nrow(x$qx), ]
  } else {
    tamanyo[1L] <- length(ages)
    qx <- qx.smooth <- qx.final <- array(NA, dim = tamanyo)
    qx[1L:nrow(x$qx), , ] <- x$qx[1L:nrow(x$qx), , ]
  }
  dimnames(qx) <- dimnames(qx.smooth) <- dimnames(qx.final) <- nombres

  # qx.smooth
  if (length(tamanyo) == 2L){
    for (jj in 1L:ncol(qx)){
      yy <- x$qx[, jj]
      if (sum(yy <= zero2NA & !is.na(yy)) > 0){
        yy[yy <= zero2NA & !is.na(yy)] <- NA
        texto <- paste0("For smoothing purposes, raw estimated death probabilities non-higher than ",
                        zero2NA, " have been replaced by 'NA'.")
        warning(texto)
      }
      yy[is.nan(yy)] <- NA
      xx <- ages[1:nrow(x$qx)]
      rango <- range_smooth(yy, nc = max.c.NA)

      fit.grad <- stats::glm(log(yy[rango[1L]:rango[2L]]) ~
                               splines::ns(xx[rango[1L]:rango[2L]], df = df, ...))
      predicc <- x$qx[, jj]
      predicc <- exp(predict(fit.grad,
                             newdata = splines::ns(xx[rango[1L]:rango[2L]], df = df, ...)))
      qx.smooth[rango[1L]:rango[2L], jj] <- predicc
    }
  } else {
    for (jj in 1L:tamanyo[2L]){
      for (kk in 1L:tamanyo[3L]){
        yy <- x$qx[, jj, kk]
        if (sum(yy <= zero2NA & !is.na(yy)) > 0){
          yy[yy <= zero2NA & !is.na(yy)] <- NA
          texto <- paste0("For smoothing purposes, raw estimated death probabilities non-higher than ",
                          zero2NA, " have been replaced by 'NA'.")
          warning(texto)
        }
        yy[is.nan(yy)] <- NA
        xx <- ages[1:nrow(x$qx)]
        rango <- range_smooth(yy, nc = max.c.NA)
        fit.grad <- stats::glm(log(yy[rango[1L]:rango[2L]]) ~
                                 splines::ns(xx[rango[1L]:rango[2L]], df = df, ...))
        predicc <- x$qx[, jj]
        predicc <- exp(predict(fit.grad,
                               newdata = splines::ns(xx[rango[1L]:rango[2L]], df = df, ...)))
        qx.smooth[rango[1L]:rango[2L], jj] <- predicc
      }
    }
  }

  if (sum(qx.smooth >= 1 & !is.na(qx.smooth)) > 0){
    qx.smooth[qx.smooth >= 1 & !is.na(qx.smooth)] <- 0.9999
    warning("Initial smoothed estimated death probabilities higher than 1 have been replaced by 0.9999.")
  }

  # qx.final
  if (threshold < ages[nrow(x$qx)]){
    if (length(tamanyo) == 2L){
      for (jj in 1L:ncol(qx)){
        yy <- qx.smooth[1L:nrow(x$qx), jj]
        tt <- as.numeric(names(which(diff(yy) < 0))) - 1
        tt <- tt[tt > threshold]
        if (length(tt) != 0){
          tt <- max(threshold, min(tt))
        } else {
          tt <- threshold
        }
        fila <- which(as.numeric(rownames(qx.smooth)) == tt)
        if (fila > (size + 1L)){
          deaths <- x$dx[(fila - size):fila, jj]
          if (length(yy) > size){
            edad <- ages[(fila - size):fila]
            yy <- qx.smooth[(fila - size):fila, jj]
            exposed <- (x$Lx[(fila - size):fila, jj] +
                          x$Lx[(fila - size - 1):(fila - 1), jj])/2
            fit.glm <- glm(cbind(deaths, ceiling(exposed) - deaths) ~ edad,
                           family = "binomial")
            pr.glm <- predict(fit.glm,
                              newdata = list(edad = ages[fila - size]:omega),
                              type = "response")
            pesos <- seq(0, 1, length.out = size + 1)
            pr.glm[1L:(size + 1L)] <- pesos*pr.glm[1L:(size + 1L)] +
              (1 - pesos)*yy
            qx.final[, jj] <- qx.smooth[, jj]
            qx.final[(fila - size):nrow(qx.final), jj] <- pr.glm
          } # End if length(yy)
        } # End if fila
      } # End for jj
    } else { # If tamanyo
      for (jj in 1L:tamanyo[2L]){
        for (kk in 1L:tamanyo[3L]){
          yy <- qx.smooth[1L:nrow(x$qx), jj, kk]
          tt <- as.numeric(names(which(diff(yy) < 0))) - 1
          tt <- tt[tt > threshold]
          if (length(tt) != 0){
            tt <- max(threshold, min(tt))
          } else {
            tt <- threshold
          }
          tt <- max(threshold, min(tt))
          fila <- which(as.numeric(rownames(qx.smooth)) == tt)
          if (fila > (size + 1L)){
            deaths <- x$dx[(fila - size):fila, jj, kk]
            if (length(yy) > size){
              edad <- ages[(fila - size):fila]
              yy <- qx.smooth[(fila - size):fila, jj, kk]
              exposed <- (x$Lx[(fila - size):fila, jj, kk] +
                            x$Lx[(fila - size - 1):(fila - 1), jj, kk])/2
              fit.glm <- glm(cbind(deaths, ceiling(exposed) - deaths) ~ edad,
                             family = "binomial")
              pr.glm <- predict(fit.glm,
                                newdata = list(edad = ages[fila - size]:omega),
                                type = "response")
              pesos <- seq(0, 1, length.out = size +1)
              pr.glm[1L:(size + 1L)] <- pesos*pr.glm[1L:(size + 1L)] +
                (1-pesos)*yy
              qx.final[, jj, kk] <- qx.smooth[, jj, kk]
              qx.final[(fila - size):nrow(qx.final), jj, kk] <- pr.glm
            } # End if length(yy)
          } # End if fila
        } # End for kk
      } # End for jj
    } # End If tamanyo
  } # End If threshold

  if (plot.explore){
    if (length(tamanyo) == 2L){
      oldpar <- graphics::par(no.readonly = TRUE)
      on.exit(graphics::par(oldpar))
      graphics::par(mfrow = c(ceiling(ncol(qx)/3), 3))
      for (jj in 1L:ncol(qx)){
        plot.explore.f(qx[, jj], qx.smooth[, jj], qx.final[, jj])
      }
      # graphics::par(mfrow = c(1, 1))
    } else {
      oldpar <- graphics::par(no.readonly = TRUE)
      on.exit(graphics::par(oldpar))
      graphics::par(mfrow = c(ceiling(tamanyo[2L]*tamanyo[3L]/3), 3))
      for (jj in 1L:tamanyo[2L]){
        for (kk in 1L:tamanyo[3L]){
          plot.explore.f(qx[, jj, kk], qx.smooth[, jj, kk], qx.final[, jj, kk])
        }
      }
      # graphics::par(mfrow = c(1, 1))
    }
  }

  return(list("qx.raw" = qx, "qx.smooth" = qx.smooth, "qx.final" = qx.final))
}




test_grad <- function(args){
  x <- args$x

  if(!(length(dim(x$qx)) %in% c(2L, 3L))){
    stop("The components of 'x' does not have the proper dimensions.")
  }

  if(!all(dim(x$qx) == dim(x$Lx) & dim(x$qx) == dim(x$dx)))
    stop("The qx, Lx and/or dx matrices/arrays in 'x' have different dimensions.")

  if(min(min(x$qx, na.rm = TRUE), min(x$Lx), min(x$dx)) < 0)
    stop("At least a value in 'qx', 'Lx' or 'dx' within 'x' is negative.")

  if ((args$df - floor(args$df)) > 0 | args$df < 0)
    stop("The argument 'df' must be a non-negative integer.")

  if ((args$threshold - floor(args$threshold)) > 0 | args$threshold < 0)
    stop("The argument 'threshold' must be a non-negative integer.")

  if ((args$size - floor(args$size)) > 0 | args$size < 0)
    stop("The argument 'size' must be a non-negative integer.")

  if ((args$omega - floor(args$omega)) > 0 | args$omega < 0)
    stop("The argument 'omega' must be a non-negative integer.")

  if (length(args$ages) != 2)
    stop("The argument 'ages' must be a vector of two components.")

  if ((args$ages[1L] - floor(args$ages[1L])) > 0 | args$ages[1L] < 0)
    stop("The argument 'ages' must be a vector of non-negative integers.")

  if ((args$ages[2L] - floor(args$ages[2L])) > 0 | args$ages[2L] < 0)
    stop("The argument 'ages' must be a vector of non-negative integers.")

  if(dim(x$qx)[1L] != (args$ages[2L] - args$ages[1L] + 1))
    stop("The number rows of the arrays within 'x' is not compatible with the values in 'ages'.")

  if(args$zero2NA < 0)
    stop("The argument 'zero2NA' cannot be negative.")

  if ((args$max.c.NA - floor(args$max.c.NA)) > 0 | args$max.c.NA < 0)
    stop("The argument 'max.c.NA' must be a non-negative integer.")
}

plot.explore.f <- function(qx, qx.s, qx.f){
  qx <- log(qx)
  qx.s <- log(qx.s)
  qx.f <- log(qx.f)
  minimo <- min(min(qx, na.rm = TRUE),
                min(qx.s, na.rm = TRUE),
                min(qx.f, na.rm = TRUE))
  maximo <- 0
  edades <- as.numeric(names(qx))
  if (sum(is.na(qx.f) == 0)){
    graphics::plot(edades, qx.f, type = "l", ylim = c(minimo, maximo),
                   col = "red", xlab = "", ylab = "")
    sel <- !is.na(qx)
    graphics::lines(edades[sel], qx.s[sel], col = "blue", lw = 1.5)
    graphics::lines(edades[sel], qx[sel], col = "black", lw = 1)
  } else {
    sel <- !is.na(qx)
    graphics::plot(edades[sel], qx.s[sel], type = "l", ylim = c(minimo, maximo),
                   col = "blue", lw = 1.5, xlab = "", ylab = "")
    graphics::lines(edades[sel], qx[sel], col = "black", lw = 1)
  }
}

range_smooth <- function(yy, nc = 3){
  # Function to determine the range to be used for smoothing
  # Maximum nc consetive missing

    es.na <- is.na(yy) # locate positions with NAs
    rle.na <- rle(es.na)
    segmentos.invalidos <- which(rle.na$values & rle.na$lengths > nc)

    if (length(segmentos.invalidos) == 0) {
      # If there are no invalid segments, return the entire range
      return(c(min(which(!es.na)), max(which(!es.na))))
    }

    # Calculate the start and end positions of invalid segments
    posiciones.invalidas <- cumsum(rle.na$lengths)[segmentos.invalidos]
    inicio.invalidos <- posiciones.invalidas - rle.na$lengths[segmentos.invalidos] + 1

    # Find the longest range between invalid segments
    max.rango <- c(1, 0)  # Inicializar rango más largo
    for (i in seq_along(inicio.invalidos)) {
      inicio <- ifelse(i == 1, 1, posiciones.invalidas[i - 1] + 1)
      fin <- inicio.invalidos[i] - 1
      if (fin - inicio > max.rango[2] - max.rango[1]) {
        max.rango <- c(inicio, fin)
      }
    }

    # Make sure to include from the last invalid segment to the end
    ultimo.fin <- posiciones.invalidas[length(posiciones.invalidas)]
    if (length(yy) - ultimo.fin > max.rango[2] - max.rango[1]) {
      max.rango <- c(ultimo.fin + 1, length(yy))
    }

    return(max.rango)
}
