create_main_htaccess_file <- function(local_basepath) {
  main_htaccess_path <- file.path(local_basepath, ".htaccess")

  # Security headers content to add
  security_headers <- c(
    "",
    "# Security Headers",
    "<IfModule mod_headers.c>",
    "    # Force HTTPS with HSTS",
    "    Header always set Strict-Transport-Security \"max-age=31536000; includeSubDomains\"",
    "    ",
    "    # Prevent MIME type sniffing",
    "    Header always set X-Content-Type-Options \"nosniff\"",
    "    ",
    "    # Prevent clickjacking",
    "    Header always set X-Frame-Options \"SAMEORIGIN\"",
    "    ",
    "    # XSS Protection",
    "    Header always set X-XSS-Protection \"1; mode=block\"",
    "    ",
    "    # Referrer Policy",
    "    Header always set Referrer-Policy \"strict-origin-when-cross-origin\"",
    "</IfModule>",
    ""
  )

  # HTTPS redirect content
  https_redirect <- c(
    "",
    "# Force HTTPS",
    "<IfModule mod_rewrite.c>",
    "    RewriteEngine On",
    "    RewriteCond %{HTTPS} off",
    "    RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]",
    "</IfModule>",
    ""
  )

  # Check if file exists
  if (file.exists(main_htaccess_path)) {
    # Read existing content
    existing_content <- readLines(main_htaccess_path, warn = FALSE)

    # Check if security headers already present
    has_security_headers <- any(grepl("Strict-Transport-Security|Security Headers", existing_content, ignore.case = TRUE))
    has_https_redirect <- any(grepl("RewriteCond.*HTTPS.*off|Force HTTPS", existing_content, ignore.case = TRUE))

    # Merge content
    new_content <- existing_content

    if (!has_security_headers) {
      new_content <- c(new_content, security_headers)
      cli::cli_alert_success("Added security headers to existing {.file .htaccess}")
    } else {
      cli::cli_alert_info("Security headers already present in {.file .htaccess}")
    }

    if (!has_https_redirect) {
      new_content <- c(new_content, https_redirect)
      cli::cli_alert_success("Added HTTPS redirect to existing {.file .htaccess}")
    } else {
      cli::cli_alert_info("HTTPS redirect already present in {.file .htaccess}")
    }

    # Write merged content
    writeLines(new_content, main_htaccess_path)
  } else {
    # Create new file with all content
    new_content <- c(
      "# Main .htaccess file with security headers",
      "# Generated by saros.base",
      https_redirect,
      security_headers
    )

    writeLines(new_content, main_htaccess_path)
    cli::cli_alert_success("Created main {.file .htaccess} with security headers at {.path {main_htaccess_path}}")
  }

  invisible(main_htaccess_path)
}

create_htaccess <-
  function(remote_basepath = "/home/",
           local_basepath = fs::path(tempdir(), "_site"),
           rel_path_base_to_parent_of_user_restricted_folder = fs::path(tempdir(), "Reports", "2022", "Mesos"),
           local_main_password_path = ".main_htpasswd_public",
           username_folder_matching_df = NULL,
           universal_usernames = c("admin"),
           log_rounds = 12,
           append_users = TRUE,
           password_input = "prompt",
           create_main_htaccess = FALSE) {
    abs_path_parents <-
      file.path(
        local_basepath,
        rel_path_base_to_parent_of_user_restricted_folder
      )

    local_subfolders_sets <- lapply(abs_path_parents, obtain_mesos_folders_from_parent_folder)
    names(local_subfolders_sets) <- rel_path_base_to_parent_of_user_restricted_folder

    lapply(seq_along(local_subfolders_sets), function(i) {
      lapply(X = seq_along(local_subfolders_sets[[i]]), function(j) {
        ### .htaccess
        folder_name <- local_subfolders_sets[[i]][j]
        user_names <-
          if (is.data.frame(username_folder_matching_df)) {
            username_folder_matching_df$username[username_folder_matching_df$folder == folder_name]
          } else {
            folder_name
          }

        content <- paste0(
          'AuthName "Saros-report access: ', folder_name,
          '"
AuthUserFile "', file.path(
            remote_basepath,
            names(local_subfolders_sets)[i],
            folder_name,
            ".htpasswd"
          ), '"
AuthType Basic
Require valid-user
AddHandler server-parsed .html'
        )
        outpath <- file.path(
          local_basepath,
          names(local_subfolders_sets)[i],
          folder_name,
          ".htaccess"
        )

        writeLines(text = content, con = outpath)

        ### .htpasswd

        credentials <- refer_main_password_file(
          x = local_main_password_path,
          usernames = unique(c(user_names, universal_usernames)),
          log_rounds = log_rounds,
          append_users = append_users,
          password_input = password_input
        )
        credentials <- data.frame(
          username = names(credentials),
          password = unname(credentials),
          row.names = NULL, stringsAsFactors = FALSE
        )

        write_htpasswd_file(
          x = credentials,
          file = file.path(
            local_basepath,
            names(local_subfolders_sets)[i],
            folder_name,
            ".htpasswd"
          ),
          header = FALSE
        )
      })
    })

    # Create main .htaccess file with security headers if requested
    if (isTRUE(create_main_htaccess)) {
      create_main_htaccess_file(local_basepath = local_basepath)
    }

    invisible()
  }
