% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/satdad_Rfunctions.R
\name{copArchimaxMevlog}
\alias{copArchimaxMevlog}
\alias{ellArchimaxMevlog}
\alias{psiArchimaxMevlog}
\alias{psiinvArchimaxMevlog}
\title{cop-ell-psi-psiinv- functions for Archimax Mevlog models.}
\usage{
copArchimaxMevlog(x, ds,  dist = "exp", dist.param = 1)
ellArchimaxMevlog(x, ds)
psiArchimaxMevlog(t, dist = "exp", dist.param = 1)
psiinvArchimaxMevlog(t, dist = "exp", dist.param = 1)
}
\arguments{
\item{x}{A vector of size \code{d} or a \code{(N.x times d)} matrix.}

\item{ds}{An object of class \code{ds}.}

\item{dist}{The underlying distribution. A character string among \code{"exp"} (the default value), \code{"gamma"} and \code{"ext"}.}

\item{dist.param}{The parameter associated with the choice \code{dist}.  If \code{dist} is \code{"exp"}, then \code{dist.param} is a postive real, the parameter of an exponential distribution. The default value is 1. If \code{dist} is \code{"gamma"}, then \code{dist.param} is a vector that concatenates the shape  and  scale parameters (in this order) of a gamma distribution.}

\item{t}{A non negative scalar or vector.}
}
\value{
When the underlying distribution \code{dist} is
\itemize{
\item{} "exp" ; For a positive \eqn{\lambda} given by \code{dist.param}, \eqn{\psi(t)=\frac{\lambda}{t+\lambda}} and \eqn{\psi^{-1}(t)=\lambda \frac{1-t}{t}}.
 \item{} "gamma" ; For positive scale \eqn{\sigma} and shape \eqn{a} given by \code{dist.param}, \eqn{\psi(t)=\frac{1}{(t+\sigma)^a}} and \eqn{\psi^{-1}(t)=\frac{t^{-1/a}-1}{\sigma}}.
 \item{} "ext" ; \eqn{\psi(t)=\exp(-t)} and \eqn{\psi^{-1}(t)=-\ln(t)}.
}

 \code{copArchimaxMevlog} returns the copula function \eqn{C(x_1,...,x_d) = \psi(\ell(\psi^{-1}(x_1),...,\psi^{-1}(x_d)))}.

 \code{ellArchimaxMevlog} returns the stable tail dependence function  \eqn{\ell(x_1,...,x_d)}.

 \code{psiArchimaxMevlog} returns the psi function  \eqn{\psi(t)}.

 \code{psiinvArchimaxMevlog} returns the psi inverse function  \eqn{\psi^{-1}(t)}.
}
\description{
Copula function, stable tail dependence function, psi function, psi inverse function for Archimax Mevlog models.
}
\details{
The tail dependence structure is set by a \code{ds} object.  See Section \bold{Value} in \code{\link[satdad]{gen.ds}}.

Turning to Archimax structures, we follow Charpentier et al. (2014). Their algorithm (4.1 of p. 124) has been applied in \code{\link[satdad]{rArchimaxMevlog}} to generate observations sampled from the copula

\eqn{C(x_1,...,x_d) = \psi(\ell(\psi^{-1}(x_1),...,\psi^{-1}(x_d)))}

when \eqn{\ell} is here the stable tail dependence function of a Mevlog model. In this package, the stdf function \eqn{\ell} is completely characterized by the \code{ds} object. See \code{\link[satdad]{ellMevlog}}.
}
\examples{

## Fix a 7-dimensional tail dependence structure
ds7 <- gen.ds(d = 7)

## Fix the parameters for the underlying distribution
(lambda <- runif(1, 0.01, 5))
(shape <- runif(1, 0.01, 5))
(scale <- runif(1, 0.01, 5))

## Fix x and t
x <- c(0.8, 0.9, 0.5, 0.8, 0.4, 0.9, 0.9)
t <- 2

## Evaluate the functions under the underlying exponential construction
copArchimaxMevlog(x = x, ds = ds7, dist = "exp", dist.param = lambda)
ellArchimaxMevlog(x = x, ds = ds7)
psiArchimaxMevlog(t = t, dist = "exp", dist.param = lambda)
psiinvArchimaxMevlog(t = t, dist = "exp", dist.param = lambda)

## Evaluate the functions under the underlying gamma construction
copArchimaxMevlog(x = x, ds = ds7, dist = "gamma", dist.param = c(shape, scale))
ellArchimaxMevlog(x = x, ds = ds7)
psiArchimaxMevlog(t = t, dist = "gamma", dist.param = c(shape, scale))
psiinvArchimaxMevlog(t = t, dist = "gamma", dist.param = c(shape, scale))

}
\references{
Charpentier, A.,   Fougères,  A.-L.,  Genest, C. and  Nešlehová, J.G. (2014)
Multivariate Archimax copulas.
\emph{Journal of Multivariate Analysis}, \bold{126}, 118--136.
}
\seealso{
\code{\link[satdad]{rArchimaxMevlog}},  \code{\link[satdad]{gen.ds}}, \code{\link[satdad]{ellMevlog}}
}
\author{
Cécile Mercadier (\code{mercadier@math.univ-lyon1.fr})
}
