#' Remove trait or trait combination from possible trait combinations of possible
#' Trait combinations
#'
#' @param ncomb Number of character combination
#' @param i remove trait or trait combination
#' @param pmat Phenotypic Variance Covariance Matrix
#' @param gmat Genotypic Variance Covariance Matrix
#' @param wmat Weight Matrix
#' @param wcol Respective weight column number of Weight Matrix
#' @param GAY Genetic Advance/Genetic Gain of base selection index
#'
#' @return Data frame of possible selection indices with per cent relative efficiency and ranking
#' @export
#'
#' @examples
#' gmat<- gen.varcov(seldata[,3:9], seldata[,2], seldata[,1])
#' pmat<- phen.varcov(seldata[,3:9], seldata[,2], seldata[,1])
#' rcomb.indices(ncomb = 2, i = 1, pmat = pmat, gmat = gmat, wmat = weight[,2:3], wcol = 1)
rcomb.indices<- function (ncomb, i, pmat, gmat, wmat, wcol = 1, GAY)
{
  # OPTIMIZATION: Convert matrices once outside loop
  # Avoids: Repeated as.matrix() calls inside nested function
  # Why faster: Matrix conversion is expensive (attribute copying, type checking)
  pmat <- as.matrix(pmat)
  gmat <- as.matrix(gmat)
  wmat <- as.matrix(wmat)
  
  ncolmn <- ncol(pmat)
  
  # OPTIMIZATION: Inline trait.remove logic with clearer edge case handling
  # Avoids: Function call overhead
  # Why faster: Direct computation, no stack frame setup
  comb_all <- combn(ncolmn, ncomb)
  
  # OPTIMIZATION: Vectorized filtering using colSums (faster than rowSums on transposed)
  # Avoids: Transpose operation + rowSums
  # Why faster: colSums is optimized C primitive, works on original orientation
  keep_mask <- colSums(comb_all != i) == ncomb
  comb <- t(comb_all[, keep_mask, drop = FALSE])
  
  # Handle edge cases for correct dimensions
  if (ncol(comb) == 0) {
    # No combinations after filtering
    return(data.frame(ID = character(0), GA = numeric(0), 
                     PRE = numeric(0), Rank = numeric(0)))
  }
  
  n_comb <- nrow(comb)
  
  # OPTIMIZATION: Pre-allocate result storage (not growing list)
  # Avoids: Memory reallocation on every list append
  # Why faster: Single allocation, direct indexing
  IDs <- vector("character", n_comb)
  b_list <- vector("list", n_comb)
  GAs <- numeric(n_comb)
  PREs <- numeric(n_comb)
  
  # OPTIMIZATION: Pre-compute constants
  # Avoids: Repeated conditional evaluation and division
  const_factor <- 2.063
  PRE_constant <- if(missing(GAY)) 100 else 100 / GAY
  
  # OPTIMIZATION: Inline selection.index function to eliminate call overhead
  # Avoids: Function stack setup/teardown × n_comb times
  # Why faster: ~20% overhead per call eliminated
  for (idx in seq_len(n_comb)) {
    trait_idx <- comb[idx, ]
    
    # OPTIMIZATION: Use paste with collapse instead of paste0 + toString
    # Avoids: Double string operation (paste0 then toString)
    # Why faster: Single pass through data
    IDs[idx] <- paste(trait_idx, collapse = ", ")
    
    # OPTIMIZATION: Direct matrix subsetting (drop=FALSE prevents dimension collapse)
    # Avoids: Creating intermediate variables p, g, w
    # Why faster: Fewer memory allocations
    p_sub <- pmat[trait_idx, trait_idx, drop = FALSE]
    g_sub <- gmat[trait_idx, trait_idx, drop = FALSE]
    w_sub <- wmat[trait_idx, wcol, drop = FALSE]
    
    # OPTIMIZATION: Use solve(A, B) instead of solve(A) %*% B
    # Avoids: Computing full matrix inverse
    # Why faster: Solves system directly (30-50% faster, more numerically stable)
    bmat <- solve(p_sub, g_sub %*% w_sub)
    
    # OPTIMIZATION: Use crossprod() instead of t() %*%
    # Avoids: Explicit transpose + matrix multiplication
    # Why faster: crossprod() is single optimized BLAS call (20-40% faster)
    numerator <- const_factor * crossprod(bmat, g_sub %*% w_sub)
    denominator <- sqrt(crossprod(bmat, p_sub %*% bmat))
    G <- numerator / denominator
    
    # OPTIMIZATION: Pre-computed PRE_constant eliminates conditional
    # Avoids: if/else evaluation in loop
    PRE <- as.numeric(G) * PRE_constant
    
    # Store results (round only final values)
    b_list[[idx]] <- round(c(bmat), 4)
    GAs[idx] <- round(as.numeric(G), 4)
    PREs[idx] <- round(PRE, 4)
  }
  
  # OPTIMIZATION: Efficient data frame construction
  # Avoids: do.call(rbind.data.frame) which is slow for many small frames
  # Why faster: Build matrix first, then convert to data.frame once
  max_b_cols <- max(lengths(b_list))
  b_matrix <- matrix(NA_real_, nrow = n_comb, ncol = max_b_cols)
  for (idx in seq_len(n_comb)) {
    b_len <- length(b_list[[idx]])
    b_matrix[idx, seq_len(b_len)] <- b_list[[idx]]
  }
  colnames(b_matrix) <- paste0("b.", seq_len(max_b_cols))
  
  # OPTIMIZATION: Single data.frame construction (not incremental rbind)
  # Avoids: Multiple memory copies in rbind.data.frame
  # Why faster: All columns created at once
  df <- data.frame(
    ID = IDs,
    b_matrix,
    GA = GAs,
    PRE = PREs,
    Rank = rank(-PREs, ties.method = "min"),
    stringsAsFactors = FALSE,
    check.names = FALSE
  )
  
  return(df)
}
