% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate.R
\name{simulate}
\alias{simulate}
\title{simulate}
\usage{
simulate(
  modelH0 = NULL,
  modelH1 = NULL,
  Sigma = NULL,
  mu = NULL,
  N = NULL,
  alpha = NULL,
  simOptions = list(nReplications = 500, minConvergenceRate = 0.75, type = "normal",
    missingVars = NULL, missingVarProp = 0, missingProp = 0, missingMechanism = "MCAR",
    nCores = 1),
  lavOptions = NULL,
  lavOptionsH1 = lavOptions,
  returnFmin = TRUE
)
}
\arguments{
\item{modelH0}{\code{lavaan} model string defining the (incorrect) analysis model.}

\item{modelH1}{\code{lavaan} model string defining the comparison model. If omitted, the saturated model is the comparison model.}

\item{Sigma}{population covariance matrix.}

\item{mu}{population means.}

\item{N}{sample size}

\item{alpha}{alpha error probability}

\item{simOptions}{a list of additional options specifying simulation details, see details.}

\item{lavOptions}{a list of additional options passed to \code{lavaan}, e. g., \code{list(estimator = 'mlm')} to request robust ML estimation}

\item{lavOptionsH1}{lavoptions when fitting \code{modelH1}. If \code{NULL}, the same as \code{lavOptions}.}

\item{returnFmin}{whether to return the mean unbiased Fmin over replications (i. e., \code{fmin_0 = fmin_hat - df/N})}
}
\value{
Returns empirical power: \code{sum(p < alpha) / nReplications} or a list (if \code{returnFmin = TRUE}) with the following components:
\item{\code{ePower}}{the empirical power.}
\item{\code{meanFmin}}{the estimated mean unbiased Fmin over replications (i. e., \code{fmin_0 = fmin_hat - df/N}).}
\item{\code{meanFminGroups}}{the estimated mean unbiased Fmin by groups given as a vector, assuming the df spread equally over groups. Therefore, \code{meanFmin != sum(meanFminGroups)}}
\item{\code{df}}{the model df.}
\item{\code{nrep}}{the number of successful replications.}
\item{\code{convergenceRate}}{the convergence rate of the H0 model.}
\item{\code{bChiSq}}{median chi-square bias of the H1 model}
\item{\code{bLambda}}{average median bias in lambda in the H1 model}
\item{\code{bPhi}}{average median bias in phi in the H1 model}
\item{\code{bPsi}}{average median bias in psi in the H1 model}
\item{\code{bBeta}}{average median bias in beta in the H1 model}
}
\description{
Estimates empirical power using a simulation approach.
}
\details{
The details of the simulation are specified in \code{simOptions}, which is a list that may have the following components:
\itemize{
\item \code{nReplications}: The targeted number of valid simulation runs, defaults to 500.
\item \code{minConvergenceRate}:  The minimum convergence rate required, defaults to .75. The maximum actual simulation runs are increased by a factor of 1/minConvergenceRate.
\item \code{type}: specifies whether the data should be generated from a population assuming multivariate normality (\code{'normal'}; the default), or based on an approach generating non-normal data (\code{'IG'}, \code{'mnonr'}, \code{'RK'}, or \code{'VM'}).
The approaches generating non-normal data require additional arguments detailed below.
\item \code{missingVars}: vector specifying the variables containing missing data (defaults to \code{NULL}).
\item \code{missingVarProp}: can be used instead of \code{missingVars}: The proportion of variables containing missing data (defaults to zero).
\item \code{missingProp}: The proportion of missingness for variables containing missing data (defaults to zero), either a single value or a vector giving the probabilities for each variable.
\item \code{missingMechanism}: The missing data mechanism, one of \code{'MCAR'} (the default), \code{'MAR'}, or \code{'NMAR'}.
\item \code{nCores}: The number of cores to use for parallel processing. Defaults to 1 (= no parallel processing). This requires the \code{doFuture} package.
}

\code{type = 'IG'} implements the independent generator approach (IG, Foldnes & Olsson, 2016) approach
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors. This requires the \code{covsim} package.

\code{type = 'mnonr'} implements the approach suggested by Qu, Liu, & Zhang (2020) and requires provision of  Mardia's multivariate skewness (\code{skewness})  and kurtosis  (\code{kurtosis}), where
skewness must be non-negative and kurtosis must be at least 1.641 skewness + p (p + 0.774), where p is the number of variables. This requires the \code{mnonr} package.

\code{type = 'RK'} implements the approach suggested by Ruscio & Kaczetow (2008) and requires provision of the population distributions
of each variable (\code{distributions}). \code{distributions} must be a list (if all variables shall be based on the same population distribution) or a list of lists.
Each component must specify the population distribution (e.g. \code{rchisq}) and additional arguments (\code{list(df = 2)}).

\code{type = 'VM'} implements the third-order polynomial method (Vale & Maurelli, 1983)
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors.

Foldnes, N. & Olsson, U. H. (2016) A Simple Simulation Technique for Nonnormal Data with Prespecified Skewness, Kurtosis, and Covariance Matrix. \emph{Multivariate Behavioral Research, 51}, 207-219. doi: 10.1080/00273171.2015.1133274

Qu, W., Liu, H., & Zhang, Z. (2020). A method of generating multivariate non-normal random numbers with desired multivariate skewness and kurtosis. \emph{Behavior Research Methods, 52}, 939-946. doi: 10.3758/s13428-019-01291-5

Ruscio, J., & Kaczetow, W. (2008). Simulating multivariate nonnormal data using an iterative algorithm. \emph{Multivariate Behavioral Research, 43}, 355-381. doi: 10.1080/00273170802285693

Vale, C. & Maurelli, V. (1983). Simulating multivariate nonnormal distributions. \emph{Psychometrika, 48}, 465-471.
}
\examples{
\dontrun{
# create Sigma and modelH0 using powerCFA
powerCFA <- semPower.powerCFA(type = 'a-priori', alpha = .05, beta = .05,
                              comparison = 'saturated',
                              Phi = .2, loadings = list(rep(.5, 3), rep(.7, 3)))
                              
# perform simulated power analysis using defaults       
simulate(modelH0 = powerCFA$modelH0, 
         Sigma = powerCFA$Sigma,
         N = powerCFA$requiredN,
         alpha = .05,
         simulatedPower = TRUE)
         

# same with additional options       
simulate(modelH0 = powerCFA$modelH0, 
         Sigma = powerCFA$Sigma,
         N = powerCFA$requiredN,
         alpha = .05,
         simulatedPower = TRUE, 
         simOptions = list(nReplications = 500, 
                           minConvergenceRate = .80, 
                           nCores = 8))


# same with IG as data generation routine
simulate(modelH0 = powerCFA$modelH0, 
         Sigma = powerCFA$Sigma,
         N = powerCFA$requiredN,
         alpha = .05,
         simulatedPower = TRUE, 
         simOptions = list(type = 'IG', 
                           skewness = c(0, 1, -2, 6, 5, 4), 
                           kurtosis = c(-3, 6, 9, 0, 2, -2)))
                           
                           
# same with mnonr as data generation routine
simulate(modelH0 = powerCFA$modelH0, 
         Sigma = powerCFA$Sigma,
         N = powerCFA$requiredN,
         alpha = .05,
         simulatedPower = TRUE, 
         simOptions = list(type = 'mnonr', 
                           skewness = 1, 
                           kurtosis = 50))
                           
                           
# same with RK as data generation routine
distributions <- list(
  list('rnorm', list(mean = 0, sd = 10)),
  list('runif', list(min = 0, max = 1)),
  list('rbeta', list(shape1 = 1, shape2 = 2)),
  list('rexp', list(rate = 1)),
  list('rpois', list(lambda = 4)),
  list('rbinom', list(size = 1, prob = .5))
)
simulate(modelH0 = powerCFA$modelH0, 
         Sigma = powerCFA$Sigma,
         N = powerCFA$requiredN,
         alpha = .05,
         simulatedPower = TRUE, 
         simOptions = list(type = 'RK', 
                           distributions = distributions))
}

}
