% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hotspot_dual_kde.R
\name{hotspot_dual_kde}
\alias{hotspot_dual_kde}
\title{Estimate the relationship between the kernel density of two layers of points}
\usage{
hotspot_dual_kde(
  x,
  y,
  cell_size = NULL,
  grid_type = "rect",
  bandwidth = NULL,
  bandwidth_adjust = 1,
  method = "ratio",
  grid = NULL,
  weights = NULL,
  transform = TRUE,
  quiet = FALSE,
  ...
)
}
\arguments{
\item{x, y}{\code{\link[sf]{sf}} data frames containing points.}

\item{cell_size}{\code{numeric} value specifying the size of each equally
spaced grid cell, using the same units (metres, degrees, etc.) as used in
the \code{sf} data frame given in the \code{x} argument. Ignored if
\code{grid} is not \code{NULL}. If this argument and \code{grid} are
\code{NULL} (the default), the cell size will be calculated automatically
(see Details).}

\item{grid_type}{\code{character} specifying whether the grid should be made
up of squares (\code{"rect"}, the default) or hexagons (\code{"hex"}).
Ignored if \code{grid} is not \code{NULL}.}

\item{bandwidth}{either a single \code{numeric} value specifying the
bandwidth to be used in calculating the kernel density estimates, or a list
of exactly 2 such values. If this argument is \code{NULL} (the default),
the bandwidth for both \code{x} and \code{y} will be determined
automatically using the result of \code{\link[MASS]{bandwidth.nrd}} called
on the co-ordinates of the points in \code{x}. If this argument is
\code{list(NULL, NULL)}, separate bandwidths will be determined
automatically for \code{x} and \code{y} based on each layer.}

\item{bandwidth_adjust}{single positive \code{numeric} value by which the
value of \code{bandwidth} for both \code{x} and \code{y} will be
multiplied, or a list of two such values. Useful for setting the bandwidth
relative to the default.}

\item{method}{\code{character} specifying the method by which the densities,
\code{d()}, of \code{x} and \code{y} will be related:
\describe{
\item{\code{ratio}}{(the default) calculates the density of \code{x}
divided by the density of \code{y}, i.e. \code{d(x) / d(y)}.}
\item{\code{log}}{calculates the natural logarithm of the density of
\code{x} divided by the density of \code{y}, i.e.
\code{log(d(x) / d(y))}.}
\item{\code{diff}}{calculates the difference between the density of
\code{x} and the density of \code{y}, i.e. \code{d(x) - d(y)}.}
\item{\code{sum}}{calculates the sum of the density of \code{x} and the
density of \code{y}, i.e. \code{d(x) + d(y)}.}
}
The result of this calculation will be returned in the \code{kde} column of
the return value.}

\item{grid}{\code{\link[sf]{sf}} data frame containing polygons, which will
be used as the grid for which densities are estimated.}

\item{weights}{\code{NULL} (the default) or a vector of length two giving
either \code{NULL} or the name of a column in each of \code{x} and
\code{y} to be used as weights for weighted counts and KDE values.}

\item{transform}{the underlying SpatialKDE package cannot calculate kernel
density for lon/lat data, so this must be transformed to use a projected
co-ordinate reference system. If this argument is \code{TRUE} (the
default) and \code{sf::st_is_longlat(x)} is \code{TRUE}, \code{x}, \code{y}
and \code{grid} (if provided) will be transformed automatically using
\code{link{st_transform_auto}} before the kernel density is estimated and
transformed back afterwards. Set this argument to \code{FALSE} to suppress
automatic transformation of the data.}

\item{quiet}{if set to \code{TRUE}, messages reporting the values of any
parameters set automatically will be suppressed. The default is
\code{FALSE}.}

\item{...}{Further arguments passed to \code{\link[SpatialKDE]{kde}}.}
}
\value{
An \code{\link[sf]{sf}} tibble of grid cells with corresponding point
counts and dual kernel density estimates for each cell. This can be plotted
using \code{\link{autoplot}}.

This function creates a regular two-dimensional grid of cells (unless a
custom grid is specified with \code{grid}), calculates the density of points
in each cell for each of \code{x} and \code{y} using functions from the
\code{SpatialKDE} package, then produces a value representing a relation
between the two densities. The count of points in each cell is also returned.

Dual kernel density values can be useful for understanding the relationship
between the distributions of two sets of point locations. For example:
\itemize{
\item The ratio between two densities representing the locations of burglaries
and the locations of houses can show the distribution of the risk
(incidence rate) of burglaries. The logged ratio may be useful to show
relationships where one set of points has an extremely skewed
distribution.
\item The difference between two densities can show the change in distributions
between two points in time.
\item The sum of two densities can be used to estimate the total density of two
types of point, e.g. the locations of occurrences of two diseases.
}
\subsection{Coverage of the output data}{

The grid produced by this function covers the convex hull of the points in
\code{x}. This means the result may include KDE values for cells that are
outside the area for which data were provided, which could be misleading. To
handle this, consider cropping the output layer to the area for which data
are available. For example, if you only have crime data for a particular
district, crop the output dataset to the district boundary using
\code{\link[sf]{st_intersection}}.
}

\subsection{Automatic cell-size selection}{

If no cell size is given then the cell size will be set so that there are 50
cells on the shorter side of the grid. If the \code{x} SF object is projected
in metres or feet, the number of cells will be adjusted upwards so that the
cell size is a multiple of 100.
}
}
\description{
Estimate the relationship between the kernel density of two layers of points
}
\examples{
# See also the examples for `hotspot_kde()` for examples of how to specify
# `cell_size`, `bandwidth`, etc.

library(sf)

# Transform data to UTM zone 15N so that cell_size and bandwidth can be set
# in metres
memphis_robberies_utm <- st_transform(memphis_robberies, 32615)
memphis_population_utm <- st_transform(memphis_population, 32615)

# Calculate burglary risk based on residential population. `weights` is set
# to `c(NULL, population)` so that the robberies layer is not weighted and
# the population layer is weighted according to the number of residents in
# each census block.
\donttest{
hotspot_dual_kde(
  memphis_robberies_utm,
  memphis_population_utm,
  bandwidth = list(NULL, NULL),
  weights = c(NULL, population)
)
}

}
\references{
Yin, P. (2020). Kernels and Density Estimation. \emph{The Geographic
Information Science & Technology Body of Knowledge} (1st Quarter 2020
Edition), John P. Wilson (ed.).
doi:\doi{10.22224/gistbok/2020.1.12}
}
