% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utilities_shrinkDSM.R
\name{prep_tvinput}
\alias{prep_tvinput}
\title{Prepare time-varying inputs for estimation of a dynamic survival model}
\usage{
prep_tvinput(
  surv_data,
  covariate_data,
  id_var,
  surv_var,
  delta_var,
  interval_var,
  covariate_id_var = id_var
)
}
\arguments{
\item{surv_data}{data frame containing meta-data for each observation (survival time and
censoring indicator) as well as an ID for each observation.}

\item{covariate_data}{data frame containing the time-varying covariates
(one per observation and time interval), an index for which time interval each covariate
is observed in as well as an ID for each observation.}

\item{id_var}{character string specifying the column name of the ID variable. If the name
is different in \code{surv_data} and \code{covariate_data}, \code{id_var} will be used in
\code{surv_data}, whereas \code{covariate_id_var} will be used in \code{covariate_data}.}

\item{surv_var}{character string specifying the column name of the survival times in
\code{surv_data}.}

\item{delta_var}{character string specifying the column name of the status indicators in
\code{surv_data}, 0 = censored or 1 = event observed..}

\item{interval_var}{character string specifying the column name of the time interval ID in
\code{covariate_data}.}

\item{covariate_id_var}{character string specifying the column name of the ID variable in
\code{covariate_data}. Defaults to \code{id_var}.}
}
\value{
Returns an object of class \code{data.frame} and \code{tvsurv} to be used as an input in
\code{shrinkDSM}.
}
\description{
This function pre-processes time-varying inputs in such a way that \code{shrinkDSM}
can work with time-varying inputs. Its main inputs are two data frames, namely
\code{surv_data} and \code{covariate_data}. \code{surv_data} contains meta data about
each observation (i.e. survival time and censoring indicator), while \code{covariate_data}
contains the time-varying covariates (one per observation and time interval) and an
index for which time interval each covariate is observed in. The two are merged
together via an ID that needs to be unique for each observation and present in both
\code{surv_data} and \code{covariate_data}.
}
\examples{
# A toy example with 5 observations and 2 covariates, observed over 3 time periods
set.seed(123)
n_obs <- 5
surv_var <- round(rgamma(n_obs, 1, .1)) + 1
delta_var <- sample(size = n_obs, c(0, 1), prob = c(0.2, 0.8), replace = TRUE)

surv_data <- data.frame(id_var = 1:n_obs, surv_var, delta_var)

# Determine intervals
S <- c(3, 11)

# Create synthetic observations for each individual
covariate_list <- list()

for (i in 1:n_obs) {
  nr_periods_survived <- sum(surv_var[i] > S) + 1
  covariate_list[[i]] <- data.frame(id_var = i,
                                    interval_var = 1:nr_periods_survived,
                                    x1 = rnorm(nr_periods_survived),
                                    x2 = rnorm(nr_periods_survived))
}

# Bind all individual covariate data frames together
# Each observation now has a covariate in each period they
# were observed in.
covariate_data <- do.call(rbind, covariate_list)

# Call prep_tvinput to pre-process for shrinkDSM
merged_data <- prep_tvinput(surv_data,
                            covariate_data,
                            id_var = "id_var",
                            surv_var = "surv_var",
                            delta_var = "delta_var",
                            interval_var = "interval_var")

# Can now be used in shrinkDSM
# Note that delta is now automatically extracted from merged_data,
# providing it will throw a warning
mod <- shrinkDSM(surv_var ~ x1 + x2, merged_data, S = S)
}
\author{
Daniel Winkler \email{daniel.winkler@wu.ac.at}

Peter Knaus \email{peter.knaus@wu.ac.at}
}
