% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simEventCox.R
\name{simEventCox}
\alias{simEventCox}
\title{Simulate Event History Data Based on Cox Models}
\usage{
simEventCox(
  N,
  cox_fits,
  L0_old,
  A0_old,
  n_event_max = c(1, 1),
  term_events = 1,
  intervention1 = NULL,
  intervention2 = NULL
)
}
\arguments{
\item{N}{Integer. The number of individuals to simulate.}

\item{cox_fits}{A named list of fitted Cox proportional hazards models (\code{coxph} objects),
one for each event type. The names are used as event type labels.}

\item{L0_old}{A vector of previously observed baseline covariate values for L0,
used for resampling baseline covariates.}

\item{A0_old}{A vector of previously observed baseline covariate values for A0,
used for resampling baseline covariates.}

\item{n_event_max}{Integer vector. Maximum number of times each event type can occur
per individual.}

\item{term_events}{Integer or integer vector. Indices of event types that are terminal,
i.e., events that stop further simulation for an individual.}

\item{intervention1}{Optional function. Takes arguments \verb{(j, sim_matrix)} and returns
an updated simulation matrix. Used to modify covariates dynamically at each event iteration.}

\item{intervention2}{Optional function. Takes arguments \verb{(j, H_j)} and returns a modified
baseline cumulative hazard vector for event type \code{j}. Allows dynamic hazard modification.}
}
\value{
A \code{data.table} with one row per event per individual containing:
\itemize{
\item \code{ID} — Individual identifier.
\item \code{Time} — Event time.
\item \code{Delta} — Event type indicator.
\item Baseline covariates \code{L0}, \code{A0}.
\item Columns for each event type indicating cumulative event counts.
}
}
\description{
Simulates recurrent and terminal event data for a cohort of individuals based on a list
of fitted Cox proportional hazards models. Each event type is governed by its own model,
and simulation proceeds by iteratively sampling event times until a terminal event occurs.
}
\details{
The function simulates individual event histories by:
\enumerate{
\item Sampling initial baseline covariates (\code{L0}, \code{A0}) by resampling observed values.
\item Extracting baseline cumulative hazard functions from the Cox models.
\item Iteratively sampling event times.
\item Updating covariate histories and event counts.
\item Stopping simulation per individual after a terminal event or maximum events reached.
}
}
\examples{
# The observed data
data_obs <- simDisease(N = 1000)
data_obs <- IntFormatData(data_obs, N_cols = 6)

# Fit Cox models
cox_death <- survival::coxph(survival::Surv(tstart, tstop, Delta == 1)
~ L0 + A0 + L, data = data_obs)
cox_Disease <- survival::coxph(survival::Surv(tstart, tstop, Delta == 2)
~ L0 + A0, data = data_obs[L == 0])

# Then simulate new data:
cox_fits <- list("D" = cox_death, "L" = cox_Disease)
new_data <- simEventCox(100, cox_fits = cox_fits, L0_old = data_obs$L0, A0_old = data_obs$A0)

}
