\name{sparsereg}
\alias{sparsereg}
\title{Sparse regression for experimental and observational data.}
\description{Function for fitting a Bayesian LASSOplus model for sparse models with uncertainty, facilitating the discovery of various types of interactions. Function takes a dependent variable, an optional matrix of (pre-treatment) covariates, and a (optional) matrix of categorical treatment variables. Includes correct calculation of uncertainty estimates, including for data with repeated observations.}
\usage{
sparsereg(y, X, treat=NULL, EM=FALSE, gibbs=200, burnin=200, thin=10,  
type="linear", scale.type="none", baseline.vec=NULL, 
id=NULL, id2=NULL, id3=NULL, save.temp=FALSE, conservative=TRUE)
}
\arguments{
  \item{y}{Dependent variable.}
\item{X}{Covariates. Typical vocabulary would refer to these as "pre-treatment" covariates.}
\item{treat}{Matrix of categorical treatment variables. May be a matrix with one column in the case of there being only one treatment variable.}
\item{EM}{Whether to fit model via EM or MCMC.  EM is much quicker, but only returns point estimates.  MCMC is slower, but returns posterior intervals and approximate confidence intervals.}
\item{gibbs}{Number of posterior samples to save.  Between each saved sample,  \env{thin} samples are drawn.}
\item{burnin}{
Number of burnin samples.   Between each burnin sample,  \env{thin} samples are drawn.
These iterations will not be included in the resulting analysis.
}
\item{thin}{
Extent of thinning of the MCMC chain.  Between each posterior sample, whether burnin or saved, \env{thin} draws are made.
}
\item{type}{
Type of regression model to fit.  Allowed types are linear or probit.
}
\item{baseline.vec}{Optional vector with one entry for each column of the treatment matrix.  Each entry gives the baseline condition for that treatment, which then during pre-processing is omitted for estimation so it serves as an excluded category.}
\item{id, id2, id3}{
Vectors the same lenght of the sample denoting clustering in the data.  In a conjoint experiment with repeated observations, these correspond with respondent IDs.  Up to three different sets of random effects are allowed.
}
\item{scale.type}{
Indicates the types of interactions that will be created and used in estimation. \env{scale.type="none"} generates no interactions and corresponds to simply running LASSOplus with no interactions between variables.
\env{scale.type="TX"} creates interactions between each X variable and each level of the treatment variables. \env{scale.type="TT"} creates interactions between each level of separate treatment variables.
\env{scale.type="TTX"} interacts each X variable with all values generated by \env{scale.type="TT"}. 
Note that users can create their own interactions of interest, select \env{scale.type="none"}, to return the sparse version of the user specified model.
}
\item{save.temp}{Whether to save intermediate output in a file named \env{temp_sparsereg}.  Useful for very long runs.}
\item{conservative}{Experimental.  If set to \env{FALSE}, the estimate is less conservative in selecting a variable.}
}
\details{
The function \env{sparsereg} allows for estimation of a broad range of sparse regressions.  The method allows for continuous, binary, and censored outcomes.  In experimental data, it can be used for subgroup analysis.  It pre-processes lower-order terms to generate higher-order interactions terms that are uncorrelated with their lower order component, with pre-processing generated through \env{scale.type}.  In observational data, it can be used in place of a standard regression, especially in the presence of a large number of variables.  The method also adjusts uncertainty estimates when there are repeated observations through using random effects.  For example, a conjoint design may have the same people make several comparisons, or a panel data regression may have multiple observations on the same unit.

The object contains the estimated posterior for all of the modeled effects, and analyzing the object is facilitated by the functions \env{plot}, \env{summary}, \env{violinplot}, and \env{difference}.
}
\value{
  \item{beta.mode}{Matrix of sparse (mode) estimates with rows equal to number of effects and columns for posterior samples.}
  \item{beta.mean}{Matrix of mean estimates with rows equal to number of effects and columns for posterior samples.  These estimates are not sparse, but they do predict better than the mode.}
  \item{beta.ci}{Matrix of effects used to calculate approximate confidence intervals.}
  \item{sigma.sq}{Vector of posterior estimate of error variance.}
  \item{X}{Matrix of covariates fit.  Includes interaction terms, depending on \env{scale.type}.}
  \item{varmat}{Matrix of showing which lower-order terms correspond with which effects.  Used in producing figures.}
  \item{baseline}{Vector of baseline categories for treatments.}
  \item{modeltype}{Type of sparsereg model fit. In this case, \env{onestage}.  Used by summary functions.}
 }
\references{
Ratkovic, Marc and Tingley, Dustin.  2015.  "Sparse Estimation with Uncertainty: Subgroup Analysis in Large Dimensional Design."  Working paper.

Egami, Naoki and Imai, Kosuke. 2015. "Causal Interaction in High-Dimension."  Working paper.
}
\seealso{\link{plot.sparsereg}, \link{summary.sparsereg}, \link{violinplot}, \link{difference}, \link{print.sparsereg}}
\examples{

\dontrun{
 set.seed(1)
 n<-500
 k<-5
 treat<-sample(c("a","b","c"),n,replace=TRUE,pr=c(.5,.25,.25))
 treat2<-sample(c("a","b","c","d"),n,replace=TRUE,pr=c(.25,.25,.25,.25))
 Sigma<-diag(k)
 Sigma[Sigma==0]<-.5
 X<-mvrnorm(n,m=rep(0,k),S=Sigma)
 y.true<-3+X[,2]*2+(treat=="a")*2 +(treat=="b")*(-2)+X[,2]*(treat=="b")*(-2)+
  X[,2]*(treat2=="c")*2
 y<-y.true+rnorm(n,sd=2)

##Fit a linear model.
s1<-sparsereg(y, X, cbind(treat,treat2), scale.type="TX")
s1.EM<-sparsereg(y, X, cbind(treat,treat2), EM=TRUE, scale.type="TX")

##Summarize results from MCMC fit
summary(s1)
plot(s1)
violinplot(s1)

##Summarize results from MCMC fit
summary(s1.EM)
plot(s1.EM)

##Extension using a baseline category
s1.base<-sparsereg(y, X, treat, scale.type="TX", baseline.vec="a")

summary(s1.base)
plot(s1.base)
violinplot(s1.base)

}

}
