#' @export

rtpd.cbm <- function(n, theta_0, dt, sigma) {
  theta <- rnorm(n,theta_0,sigma*sqrt(dt))%%(2*pi)
  return(theta)
}

#' @title Simulate circular Brownian motion
#' @name rtraj.cbm
#' @description `rtraj.cbm` returns a simulated path of a circular Brownian motion for given parameters
#' @usage rtraj.cbm(n, theta_0, dt, sigma, burnin=1000)
#' @details Let \eqn{\theta_t} evolve according to a circular Brownian motion given by,
#' \deqn{d\theta_t=\sigma dW_t}
#'
#' We simulate \eqn{\theta_t} by simulating from its transition density.
#'
#' @param n number of steps in the simulated path
#' @param dt Time step
#' @param theta_0 initial point
#' @param sigma volatility parameter
#' @param burnin number of initial samples to be rejected (Default is 1000)
#' @return A vector of length `n` of the simulated path from circular Brownian motion

rtraj.cbm <- function(n, theta_0, dt, sigma,burnin=1000) {
  a <- rep(theta_0, n)
  for (i in 2:n) {
    aa <- rtpd.cbm(burnin, a[i - 1], dt, sigma)
    a[i] <- aa[burnin]
  }
  return(a)
}
