% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/proposal_builder.R
\name{build_proposal}
\alias{build_proposal}
\title{Build Proposal}
\usage{
build_proposal(
  f,
  modes,
  lower = -Inf,
  upper = Inf,
  h = NULL,
  h_prime = NULL,
  steps = NULL,
  proposal_range = NULL,
  theta = 0.1,
  target_sample_size = 1000,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{f}{A function which returns the (unnormalised) probability density function of the target distribution.
The first argument must be the value at which the pdf is to be evaluated.
Additional arguments may be parameters of the distribution, which should be specified by name in the \code{...} arguments.}

\item{modes}{Numeric vector of modes of the density function.}

\item{lower}{Numeric scalar representing the lower bound of the target density.
Default is \code{-Inf} for unbounded lower support.}

\item{upper}{Numeric scalar representing the upper bound of the target density.
Default is \code{Inf} for unbounded upper support.}

\item{h}{An optional function which returns the (unnormalised) log-probability density function of the target distribution.
As for \code{f} the first argument must be the value at which the log-pdf is to be evaluated and additional parameters may be named arguments passed to \code{...}.}

\item{h_prime}{An optional function which returns the first derivative of the (unnormalised) log-probability density function of the target distribution.
As for \code{f} the first argument must be the value at which the log-pdf is to be evaluated and additional parameters may be named arguments passed to \code{...}.}

\item{steps}{Optional integer scalar specifying the number of steps in the step optimised part of the proposal density and squeezing function.}

\item{proposal_range}{Optional numeric vector of length 2 specifying the lower and upper range of the steps in the step optimised part of the proposal density and squeezing function.
This range should be contained within the interval defined by \code{lower} and \code{upper}.}

\item{theta}{Optional numeric scalar (between 0.1 and 1) defining the pre-acceptance threshold.
This dictates when no further steps should be added in the step optimised part of the proposal density and squeezing function, based on the probability of pre-acceptance.}

\item{target_sample_size}{Integer scalar indicating the typical sample size that will be requested when sampling from this density using build_sampler.
The proposal optimization process bases benchmark timings on this target size in order to select a proposal best suited to the desired sample size.
Note this does \emph{not} limit sampling to this number, it is merely a guide should the user be aware that a certain sample size will be most commonly sampled.}

\item{verbose}{Logical scalar.
If \code{TRUE}, a table detailing the optimization areas and steps will be displayed during proposal optimization.
Defaults to \code{FALSE}.}

\item{...}{Further arguments to be passed to \code{f}, \code{h}, and \code{h_prime}, if they depend on additional parameters.}
}
\value{
This returns a list which is used to construct the sampler by passing to \code{\link{build_sampler}} function.

A list containing the optimized proposal and related parameters for the specified built-in distribution:
\describe{
\item{\code{data}}{A data frame with detailed information about the proposal steps, including:
\describe{
\item{\code{x}}{The start point of each step on the x-axis.}
\item{\code{s_upper}}{The height of each step on the y-axis.}
\item{\code{p_a}}{Pre-acceptance probability for each step.}
\item{\code{s_upper_lower}}{A vector used to scale the uniform random number when the sample is accepted.}
}}
\item{\code{areas}}{A numeric vector containing the areas under:
\describe{
\item{\code{left_tail}}{The left tail bound.}
\item{\code{steps}}{The middle steps.}
\item{\code{right_tail}}{The right tail bound.}
}}
\item{\code{steps_number}}{An integer specifying the number of steps in the proposal.}
\item{\code{sampling_probabilities}}{A numeric vector with:
\describe{
\item{\code{left_tail}}{The probability of sampling from the left tail.}
\item{\code{left_and_middle}}{The combined probability of sampling from the left tail and middle steps.}
}}
\item{\code{unif_scaler}}{A numeric scalar, the inverse probability of sampling from the steps part of the proposal (\eqn{\frac{1}{p(lower < x < upper)}}). Used for scaling uniform random values.}
\item{\code{lt_properties}}{A numeric vector of 5 values required for Adaptive Rejection Sampling (ARS) in the left tail.}
\item{\code{rt_properties}}{A numeric vector of 6 values required for ARS in the right tail.}
\item{\code{alpha}}{A numeric scalar representing the uniform step area.}
\item{\code{tails_method}}{A string, either \code{"ARS"} (Adaptive Rejection Sampling) or \code{"IT"} (Inverse Transform), indicating the sampling method for the tails.}
\item{\code{proposal_bounds}}{A numeric vector specifying the left and right bounds of the target density.}
\item{\code{cnum}}{An integer representing the cache number of the created proposal in memory.}
\item{\code{symmetric}}{A numeric scalar indicating the symmetry point of the proposal, or \code{NULL} if not symmetric.}
\item{\code{f_params}}{A list of parameters for the target density that the proposal is designed for.}
\item{\code{is_symmetric}}{A logical value indicating whether the proposal is symmetric.}
\item{\code{proposal_type}}{A string indicating the type of the generated proposal:
\describe{
\item{\code{"scaled"}}{The proposal is "scalable" and standardized with \code{rate = 1}. This is used when parameter \code{rate} is either \code{NULL} or not provided. Scalable proposals are compatible with \code{\link{srexp}}.}
\item{\code{"custom"}}{The proposal is "custom" when \code{rate} is provided. Custom proposals are compatible with \code{\link{srexp_custom}}.}
}}
\item{\code{target_function_area}}{A numeric scalar estimating the area of the target distribution.}
\item{\code{dens_func}}{A string containing the hardcoded density function.}
\item{\code{density_name}}{A string specifying the name of the target density distribution.}
\item{\code{lock}}{An identifier used for saving and loading the proposal from disk.}
}
}
\description{
Constructs the step optimized proposal density, squeezing function, and log-linear tail proposal for a user defined probability density function.
}
\details{
This function is the starting point when a user wishes to build a custom sampler using StORS.
It is responsible for generating the step optimized proposal density, squeezing function, and log-linear tail proposal that can be utilized for this purpose.
The minimum information that must be supplied by the user is:
\itemize{
\item The (closed) interval of support for the distribution, [\code{lower}, \code{upper}] \eqn{\in \mathbb{R}}, which may also be half-closed on either side, or all of \eqn{\mathbb{R}}.
\item The probability density function (pdf), which need not be normalised, \code{f}.
\item Any modes of the pdf, as vector \code{modes}.
}

Optionally, the log-pdf and derivative of the log-pdf may be supplied.

\strong{Arguments for pdf}

The pdf (and log-pdf and first derivative of the log-pdf) may depend on certain parameters.
If so, these can be from the second argument onward in \code{f}.
For instance, consider the Kumaraswamy distribution, which has pdf:
\deqn{f(x; a,b) = a b x^{a-1}{ (1-x^a)}^{b-1},  \ \ \mbox{where} \ \ x \in (0,1)}
This pdf has known modes.

Then, to implement as a custom StORS sampler, we would first define the pdf in R:

\code{dkumaraswamy <- function(x, a, b) a*b*(x^(a-1))*(1-x^a)^(b-1)}

Then, to construct a StORS proposal for \eqn{a=2} and \eqn{b=2}, we would call

\code{Proposal <- build_Proposal(lower = 0, upper = 1, modes = sqrt(1/3), f = dkumaraswamy, a = 2, b = 2)}

\strong{StORS proposal construction}

StORS defines an unnormalised piecewise constant proposal density and squeezing function, with a Proposal defining the change points.
To optimise the execution speed on modern CPUs, the unnormalised piecewise constant proposal has fixed area for each segment with one end of the segment coinciding with the user's pdf.
That is, each step of the function has width defined by \eqn{w_i = (x_i - x_{i-1})} and a height determined by \eqn{h_i = \max(f(x_{i-1}), f(x_i))}, such that \eqn{w_i h_i = \alpha \ \forall\,i} where \eqn{\alpha} is constant.

Once the user has constructed the proposal, the sampling function can be built using \code{\link[=build_sampler]{build_sampler()}}.

\strong{Internal details}

The function \code{build_final_Proposal()} manages the construction of these steps and calculates values critical for the sampling process.
When the resultant Proposal is used with the \code{build_sampler()} function, these values are cached,
significantly enhancing the computational efficiency and hence improving sampling speed.
During the optimization process, we aim for a certain Proposal
size based on L1-3 memory cache size. Therefore, we test the speed of Proposals of sizes \eqn{2^m} Kb.
To achieve this, we estimate the uniform step area
based on a certain steps number that leads to the target cache size,
\eqn{ \alpha = \frac{1}{\text{number of steps}} }.

The speed testing for each possible Proposal is initially based on a sample size of 1000.
However, if the user wishes to optimize the Proposal for a different sample size, they can do so
by specifying the desired sample size using the \code{target_sample_size} argument.

In case the user wants to select a specific number of steps for the proposal
and bypass the optimization process, this can be done by specifying a steps number greater than the number of modes by 2 using the \code{steps} argument.
If the target density is heavy-tailed,
and the user wishes to stop the Proposal building process at a certain pre-acceptance threshold, this can be achieved by setting
the acceptance probability threshold \code{theta} \eqn{\theta}.
Once the steps reach this level of pre-acceptance probability,
the step construction will end \eqn{ \frac{\min(f(x_i), f(x_{i+1}))}{\max(f(x_i), f(x_{i+1}))} < \theta }.
Alternatively, if the user wishes to create the steps within certain limits on the
x-axis, they can do so by specifying the proposal limits using the \code{proposal_range} argument.
}
\examples{

# Example 1: Building a proposal for Standard Normal Distribution
# This example demonstrates constructing a proposal for a standard normal distribution
# \( f(x) \sim \mathcal{N}(0,1) \),
# and shows the optimization table by setting \code{verbose} to \code{TRUE}.

# Define the density function, its logarithm,
# and its derivative for the standard normal distribution
modes_norm = 0
f_norm <- function(x) { 1 / sqrt(2 * pi) * exp(-0.5 * x^2) }
h_norm <- function(x) { log(f_norm(x)) }
h_prime_norm <- function(x) { -x }

# Following example takes slightly too long to run on CRAN.
\donttest{
# Build the proposal for the standard normal distribution
norm_proposal = build_proposal(lower = -Inf, upper = Inf, mode = modes_norm,
 f = f_norm, h = h_norm, h_prime = h_prime_norm, verbose = TRUE)

# Plot the generated proposal
plot(norm_proposal)
}

# Example 2: proposal for a Bimodal Distribution
# This example shows how to build a proposal for sampling from a bimodal distribution,
#combining two normal distributions
# \( f(x) = 0.5 \cdot w_1(x) + 0.5 \cdot w_2(x) \),
# where \( w_1(x) \sim \mathcal{N}(0, 1) \) and \( w_2(x) \sim \mathcal{N}(4, 1) \).

# Define the bimodal density function
f_bimodal <- function(x) {
  0.5 * (1 / sqrt(2 * pi)) * exp(-(x^2) / 2) + 0.5 * (1 / sqrt(2 * pi)) * exp(-((x - 4)^2) / 2)
}
modes_bimodal = c(0.00134865, 3.99865)

# Build the proposal for the bimodal distribution
bimodal_proposal = build_proposal( f = f_bimodal, lower = -Inf, upper = Inf, mode = modes_bimodal)

# Print and plot the bimodal proposal
print(bimodal_proposal)
plot(bimodal_proposal)

# Example 3: Proposal with 500 Steps for Bimodal Distribution
# This example demonstrates constructing a proposal with 500 steps,
# for the bimodal distribution used in Example 2.

bimodal_proposal_500 = build_proposal(f = f_bimodal, lower = -Inf, upper = Inf,
 mode = modes_bimodal, steps = 500)

# Print and plot the proposal with 500 steps
print(bimodal_proposal_500)

}
\seealso{
\code{\link{build_sampler}}: Function to build and return a sampling function based on the provided proposal properties.
}
