% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DST_SlidingWindow.R
\name{DST_SlidingWindow}
\alias{DST_SlidingWindow}
\alias{update.DST_SlidingWindow}
\alias{predict.DST_SlidingWindow}
\title{DST_SlidingWindow -- Call R Functions on a Sliding Window}
\usage{
DST_SlidingWindow(f, window, rebuild, ...)

\method{update}{DST_SlidingWindow}(
  object,
  dsd,
  n = 1L,
  return = c("nothing", "model"),
  rebuild = FALSE,
  ...
)

\method{predict}{DST_SlidingWindow}(object, newdata, ...)
}
\arguments{
\item{f}{the function to be called.}

\item{window}{size of the sliding window.}

\item{rebuild}{logical; perform a rebuild after the update.}

\item{...}{additional parameters passed on to the \code{predict()} function of the underlying model.}

\item{object}{the updated \code{DST_SlidingWindow} object.}

\item{dsd}{A \link{DSD} object with the data stream.}

\item{n}{number of points from \code{dsd} to use for the update.}

\item{return}{a character string indicating what update returns. The default is \code{"nothing"}
and \code{"model"} returns the aggregated data.}

\item{newdata}{dataframe with the new data.}
}
\value{
An object of class \code{DST_SlidingWindow}.
}
\description{
Keeps a sliding window of the data stream an calls a function at regular
intervals with the contents of the window.
}
\details{
Keeps a sliding window of the data stream an calls a function at regular
intervals with the contents of the window.
The function needs to have the form

\code{f <- function(data, ...) {...}}

where \code{data} is the data.frame
with the points in the sliding window (See \code{get_points()} in \link{DSAggregate_Window}). The
function will be executed at regular intervals after \code{update()} was called with fixed
number of points. \code{update(..., rebuild = TRUE)} can be used to force recomputing the function.
This can be used with \code{n = 0} to recompute it even without adding more points.
The last valid function result can be retrieved/

Many modelling functions provide a formula interface which lets them be directly used
inside a \code{DST_SlidingWindow} (see Examples section).

If the function returns a model that supports \code{predict()}, then predict can directly be
called on the \code{DST_SlidingWindow} object.
}
\examples{
library(stream)

# create a data stream for the iris dataset
data <- iris[sample(nrow(iris)), ]
stream <- DSD_Memory(data)
stream

## Example 1: Use a function on the sliding window
summarizer <- function(data) summary(data)

s <- DST_SlidingWindow(summarizer,
  window = 100, rebuild = 50)
s

# update window with 49 points. The function is not yet called
update(s, stream, 49)
get_model(s)

# updating with the 50th point will trigger a function call (see rebuild parameter)
# note that the window is only 1/2 full and we have 50 NAs
update(s, stream, 1)
get_model(s)

# 50 more points and the function will be recomputed
update(s, stream, 50)
get_model(s)


## Example 2: Use classifier on the sliding window
reset_stream(stream)

# rpart, like most models in R, already have a formula interface that uses a
# data parameter. We can use these types of models directly
library(rpart)
cl <- DST_SlidingWindow(
  rpart, formula = Species ~ Petal.Length + Petal.Width,
  window = 100, rebuild = 50)
cl

# update window with 50 points so the model is built
update(cl, stream, 50)
get_model(cl)

# update with 40 more points and force the function to be recomputed even though it would take
#  50 points to automatically rebuild.
update(cl, stream, 40, rebuild = TRUE)
get_model(cl)

# rpart supports predict, so we can use it directly with the DST_SlidingWindow
new_points <- get_points(stream, n = 5)
predict(cl, new_points, type = "class")

## Example 3: Regression using a sliding window
reset_stream(stream)

## lm can be directly used
reg <- DST_SlidingWindow(
  lm, formula = Sepal.Length ~ Petal.Width + Petal.Length,
  window = 100, rebuild = 50)
reg

update(reg, stream, 100)
get_model(reg)

# lm supports predict, so we can use it directly with the DST_SlidingWindow
new_points <- get_points(stream, n = 5)
predict(reg, new_points)
}
\seealso{
Other DST: 
\code{\link{DSAggregate}()},
\code{\link{DSC}()},
\code{\link{DSClassifier}()},
\code{\link{DSOutlier}()},
\code{\link{DSRegressor}()},
\code{\link{DST}()},
\code{\link{DST_WriteStream}()},
\code{\link{evaluate}},
\code{\link{predict}()},
\code{\link{stream_pipeline}},
\code{\link{update}()}
}
\author{
Michael Hahsler
}
\concept{DST}
