% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vim.R
\name{vim}
\alias{vim}
\title{Estimate AUC VIM}
\usage{
vim(
  type,
  time,
  event,
  X,
  landmark_times = stats::quantile(time[event == 1], probs = c(0.25, 0.5, 0.75)),
  restriction_time = max(time[event == 1]),
  approx_times = NULL,
  large_feature_vector,
  small_feature_vector,
  conditional_surv_preds = NULL,
  large_oracle_preds = NULL,
  small_oracle_preds = NULL,
  conditional_surv_generator = NULL,
  conditional_surv_generator_control = NULL,
  large_oracle_generator = NULL,
  large_oracle_generator_control = NULL,
  small_oracle_generator = NULL,
  small_oracle_generator_control = NULL,
  cf_folds = NULL,
  cf_fold_num = 5,
  sample_split = TRUE,
  ss_folds = NULL,
  robust = TRUE,
  scale_est = FALSE,
  alpha = 0.05,
  verbose = FALSE
)
}
\arguments{
\item{type}{Type of VIM to compute. Options include \code{"accuracy"}, \code{"AUC"}, \code{"Brier"}, \code{"R-squared"}
\code{"C-index"}, and \code{"survival_time_MSE"}.}

\item{time}{\code{n x 1} numeric vector of observed
follow-up times. If there is censoring, these are the minimum of the
event and censoring times.}

\item{event}{\code{n x 1} numeric vector of status indicators of
whether an event was observed.}

\item{X}{\code{n x p} data.frame of observed covariate values}

\item{landmark_times}{Numeric vector of length J1 giving
landmark times at which to estimate VIM (\code{"accuracy"}, \code{"AUC"}, \code{"Brier"}, \code{"R-squared"}).}

\item{restriction_time}{Maximum follow-up time for calculation of \code{"C-index"} and \code{"survival_time_MSE"}.}

\item{approx_times}{Numeric vector of length J2 giving times at which to
approximate integrals. Defaults to a grid of 100 timepoints, evenly spaced on the quantile scale of the distribution of observed event times.}

\item{large_feature_vector}{Numeric vector giving indices of features to include in the 'large' prediction model.}

\item{small_feature_vector}{Numeric vector giving indices of features to include in the 'small' prediction model. Must be a
subset of \code{large_feature_vector}.}

\item{conditional_surv_preds}{User-provided estimates of the conditional survival functions of the event and censoring
variables given the full covariate vector (if not using the \code{vim()} function to compute these nuisance estimates).
Must be a named list of lists with elements \code{S_hat}, \code{S_hat_train}, \code{G_hat}, and \code{G_hat_train}. Each of these is itself
a list of length \code{K}, where \code{K} is the number of cross-fitting folds. Each element of these lists is a matrix with J2 columns and number of rows
equal to either the number of samples in the \code{k}th fold (for \code{S_hat} or \code{G_hat}) or the number of samples used to compute the nuisance estimator
for the \code{k}th fold.}

\item{large_oracle_preds}{User-provided estimates of the oracle prediction function using \code{large_feature_vector}. Must be a named list of lists
with elements \code{f_hat} and \code{f_hat_train}. Each of these is itself a list of length \code{K}. Each element of these lists is a matrix with J1 columns
(for landmark time VIMs) or 1 column (for \code{"C-index"} and \code{"survival_time_MSE"}).}

\item{small_oracle_preds}{User-provided estimates of the oracle prediction function using \code{small_feature_vector}. Must be a named list of lists
with elements \code{f_hat} and \code{f_hat_train}. Each of these is itself a list of length \code{K}. Each element of these lists is a matrix with J1 columns
(for landmark time VIMs) or 1 column (for \code{"C-index"} and \code{"survival_time_MSE"}).}

\item{conditional_surv_generator}{A user-written function to estimate the conditional survival functions of the event and censoring variables. Must take arguments
\code{time}, \code{event}, \code{folds} (cross-fitting fold identifiers), and
\code{newtimes} (times at which to generate predictions).}

\item{conditional_surv_generator_control}{A list of arguments to pass to \code{conditional_surv_generator}.}

\item{large_oracle_generator}{A user-written function to estimate the oracle prediction function using \code{large_feature_vector}.Must take arguments
\code{time}, \code{event}, and \code{folds} (cross-fitting fold identifiers).}

\item{large_oracle_generator_control}{A list of arguments to pass to \code{large_oracle_generator}.}

\item{small_oracle_generator}{A user-written function to estimate the oracle prediction function using \code{small_feature_vector}.Must take arguments
\code{time}, \code{event}, and \code{folds} (cross-fitting fold identifiers).}

\item{small_oracle_generator_control}{A list of arguments to pass to \code{small_oracle_generator}.}

\item{cf_folds}{Numeric vector of length \code{n} giving cross-fitting folds}

\item{cf_fold_num}{The number of cross-fitting folds, if not providing \code{cf_folds}}

\item{sample_split}{Logical indicating whether or not to sample split}

\item{ss_folds}{Numeric vector of length \code{n} giving sample-splitting folds}

\item{robust}{Logical, whether or not to use the doubly-robust debiasing approach. This option
is meant for illustration purposes only --- it should be left as \code{TRUE}.}

\item{scale_est}{Logical, whether or not to force the VIM estimate to be nonnegative}

\item{alpha}{The level at which to compute confidence intervals and hypothesis tests. Defaults to 0.05}

\item{verbose}{Whether to print progress messages.}
}
\value{
Named list with the following elements:
\item{result}{Data frame giving results. See the documentation of the individual \code{vim_*} functions for details.}
\item{folds}{A named list giving the cross-fitting fold IDs (\code{cf_folds}) and sample-splitting fold IDs (\code{ss_folds}).}
\item{approx_times}{A vector of times used to approximate integrals appearing in the form of the VIM estimator.}
\item{conditional_surv_preds}{A named list containing the estimated conditional event and censoring survival functions.}
\item{large_oracle_preds}{A named list containing the estimated large oracle prediction function.}
\item{small_oracle_preds}{A named list containing the estimated small oracle prediction function.}
}
\description{
Estimate AUC VIM
}
\examples{
# This is a small simulation example
set.seed(123)
n <- 100
X <- data.frame(X1 = rnorm(n), X2 = rbinom(n, size = 1, prob = 0.5))

T <- rexp(n, rate = exp(-2 + X[,1] - X[,2] + .5 *  X[,1] * X[,2]))

C <- rexp(n, exp(-2 -.5 * X[,1] - .25 * X[,2] + .5 * X[,1] * X[,2]))
C[C > 15] <- 15

time <- pmin(T, C)
event <- as.numeric(T <= C)

# landmark times for AUC
landmark_times <- c(3)

output <- vim(type = "AUC",
              time = time,
              event = event,
              X = X,
              landmark_times = landmark_times,
              large_feature_vector = 1:2,
              small_feature_vector = 2,
              conditional_surv_generator_control = list(SL.library = c("SL.mean", "SL.glm")),
              large_oracle_generator_control = list(SL.library = c("SL.mean", "SL.glm")),
              small_oracle_generator_control = list(SL.library = c("SL.mean", "SL.glm")),
              cf_fold_num = 2,
              sample_split = FALSE,
              scale_est = TRUE)

print(output$result)
}
\seealso{
\link{vim_accuracy} \link{vim_AUC} \link{vim_brier} \link{vim_cindex} \link{vim_rsquared} \link{vim_survival_time_mse}
}
