#' Conditional probability given the observed data estimate
#'
#' @description {This is the base function of the package that uses the Beran nonparametric conditional survival function
#' estimator for right-censored data to estimate the conditional probability of the
#' event \code{T<t} (i.e., the event occurring before time t) or \code{T>t} (i.e., the event occurring
#' after time t), given the observed data.
#' }
#'
#' @usage Csurv(Y, M, censor, t, h=NULL, ktype="gaussian")
#'
#' @param Y The numeric vector of event-time or observed time.
#' @param M The numeric vector of marker value.
#' @param censor The censoring indicator, \code{1} if event, \code{0} otherwise.
#' @param t A scalar time point used to calculate the the ROC curve
#' @param h A scalar bandwidth value for kernel weights estimation. The default is the value obtained using the method of Sheather and Jones (1991).
#' @param ktype A character string specifying the desired kernel needed for Beran weight calculation. The possible options are "\code{normal}", "\code{epanechnikov}", "\code{tricube}", "\code{boxcar}", "\code{triangular}", or "\code{quartic}". The defaults is "\code{gaussian}" kernel density.
#'
#' @return Return a list containing:
#'
#' \item{positive }{estimate of  \code{P(T<t|Y,censor,M)}.}
#'
#' \item{negative }{estimate of  \code{P(T>t|Y,censor,M)}.}
#'
#' @examples library(tdPRC);
#' data(mayo);
#'
#' data <- mayo[ ,c( "time","censor","mayoscore5" )] ;
#' t <- 365*6;
#'
#' est <- Csurv(Y=data$time, M=data$mayoscore5, censor=data$censor, t=t, ktype="gaussian")
#'
#' summary(est$positive)
#'
#' @references Beyene, K.M., Chen, D.G., and Kifle, Y.G. (2024). A novel nonparametric time‐dependent precision-recall curve estimator for right‐censored survival data. \emph{Biometrical Journal}, 66(3), 2300135.
#' @references Beyene, K.M. and El Ghouch A. (2020). Smoothed time-dependent receiver operating characteristic curve for right censored survival data. \emph{Statistics in Medicine}, 39: 3373-3396.
#'
#' @export

Csurv <- function( Y, M, censor, t, h=NULL, ktype="gaussian") {
  if (is.null(h)) {
    h <- bw.SJ(M, method = "dpi")
  }
  if(ktype=="normal"){
    ktype <- "gaussian"
  }
  n <- length(M) ;
  positive <- rep(NA, n);
  for (i in 1:n) {
    if (Y[i] > t) {
      positive[i] <- 0;
    } else {
      if (censor[i] == 1) {
        positive[i] <- 1;
      } else {
        St <- Beran(time = Y, status = censor, covariate = M, x = M[i], y = t, kernel = ktype, bw = h)
        Sy <- Beran(time = Y, status = censor, covariate = M, x = M[i], y = Y[i], kernel = ktype, bw = h)
       if (Sy == 0 ) {
          positive[i] <- 1 ;
        } else {
          positive[i] <- 1 - St/Sy;
        }
      }
    }
  }
  negative <- 1 - positive ;
  return(list(positive = positive, negative = negative));
}

