% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/typehint.r
\name{check_types}
\alias{check_types}
\title{Type hints - Automatic checks of function arguments}
\usage{
check_types(show.msg = TRUE, abort = TRUE, messages = NULL, color = "#bd0245")
}
\arguments{
\item{show.msg}{Indicates if a message is shown whenever a check fails
(default is \code{TRUE}).}

\item{abort}{Indicates if checks are stopped after the first error occurred
(default is \code{TRUE}), or if all checks are performed.}

\item{messages}{A vector with five message templates to be used as error
messages. \code{NULL}, if the default templates shall be used. Templates can
make use of predefined placeholders to convey information important for
understanding the source of the problem. See below for a comprehensive
discussion of error messages.}

\item{color}{Standard hex RGB color code of the error messages (default is
\code{"#bd0245"}).}
}
\value{
\code{TRUE}, if \emph{all} parameter values provided in the function
call pass \emph{all} tests / adhere to all restrictions defined in the type
hint commments, \code{FALSE} otherwise.
}
\description{
The \code{check_types()} function is used within the body of a
function to evaluate the parameters of a call to that function against the
requirements defined in the type hint comments. See \emph{Details} section
for more information on type hint comments.
}
\section{How do type hints work? Overview.}{


Type hints are special comments with a leading \code{#|} within a function
body indicating the intended nature of the function's arguments in terms of
data types, dimensions and even permitted values. The actual parameters with
which the function is called can be evaluated against these type hint
comments using the \code{check_types()} function.

\code{check_types()} returns \code{FALSE} if any of the checks fails.
Checking can be aborted after the first error occurs, or it can be continued
until all checks have been performed. Optionally, the user is shown a
message indicating the nature of the problem with the function arguments.
The messages can be completely customized using placerholder variables for
all kinds of relevant information.
}

\section{Type hint comments}{

\subsection{Placement of comments}{

Type hint comments always need to be placed \emph{inside} a function body
and refer to the arguments of that function. They can be placed
\emph{anywhere} in the function body (even after the call of
\code{check_types()}). Type hint comments are regular R comments but start
with \code{#|} (hash plus pipe, without blanks in between). Each function
argument will have its own type hint comment line. Type hint comments can
cover a subset of all arguments, so there can be arguments without any type
hint restrictions.
}

\subsection{Comment syntax}{

Type hint comments consist of a data type check and (optionally) dimension
and value checks: \itemize{

\item \bold{Data type check}: The data type checks for the data type of the
argument. At this point, the data type check needs to be the first check in
a type hint comment and can only comprise one permitted data type. The
syntax is \code{argument_name data_type}. A valid type hint comment
consisting only of a data type check could thus look like this: \code{#|
 degrees_celsius numeric}.

\item \bold{Dimension check}: The dimension check checks for the number and
size of the dimensions of the argument. It is constructed using the
\code{dim()} function. \code{dim()} takes one parameter per dimension of the
argument. The parameters specify the size of each of the dimensions of the
argument either as specific values or as comparisons. So, the general syntax
is: \code{dim([comparison_operator]dimsize [,[comparison_operator]#'
 dim_size]*)}. For example, if the argument (called \code{unemployment}) is
required to be a dataframe with exactly four columns and at least two rows
then the type hint comment would look like this: \code{#| unemployment
 data.frame dim(>=2, 4)}. When \code{check_types()} evaluates the parameters
supplied in the function call it looks for the number of dimensions of the
parameter as well as the size of each dimension.

\item \bold{Value check}: The value check evaluates the actual value of the
parameter supplied in the function call and rejects the value if it is on an
exclude list. Such exclude lists are constructed using the \code{not()}
function. The \code{not()} function expects as its arguments the values that
shall not be permitted as parameter values. These values can include
\code{NA} and \code{NULL}. The general syntax of the  \code{not()} function
is: \code{not(excludevalue[,excludevalue]*)}. If we had an argument called
\code{surname} and this argument must not be \code{NA} or\code{""} (empty
character) then the required type hint check would like this: \code{#|
 surname character not("", NA)}. There can be multiple \code{not}-lists in
each type hint comment. If the parameter supplied in the function call
consists of, by its nature, multiple elements, like it is the case with
dataframes, list, and matrices, then the value check fails if \emph{any}
element of the parameter provided in the function call is on the exclude
list.}

When formulating \code{dim} or \code{not} restrictions you can use the
values of other parameters of the function call. So, if you have a function
with two arguments, a number of children (\code{num.children}) and a numeric
vector with the ages of these children (\code{age.children}) you can have a
type hint comment for the latter which looks like this: \code{#|
 age.children numeric dim(num.children)}.
}
}

\section{If a type hint check fails}{


If any of the checks fails \code{check_types()} returns \code{FALSE},
otherwise it returns \code{TRUE}. If \code{show.msg=TRUE} then a message
will be shown in the R console. The messages can be customized using
templates (see next section). Depending on the value of \code{abort} the
checking of parameters is continued (\code{abort=FALSE}) or stopped
immediately (\code{abort=TRUE}), i.e. no further checks are performed after
the first error.
}

\section{Type hint output messages}{

\subsection{Message templates}{

The error messages that are shown (if \code{show.msg=TRUE}) when a check
fails are based on templates. The templates are provided to the
\code{check_types()} function via the \code{messages} argument.
\code{messages} is a character vector with five elements, one for each
possible kind of error message (or \code{NULL}, if the default error
messages shall be used); the types of error messages are:

\itemize{

\item General intro message (default: \code{"Problem in function '#fun()'"})

\item Wrong parameter type (default: \code{"Argument '#arg' (#argval) is of
 class #type_is but needs to be of class #type_req."})

\item Wrong dimension size of parameter (default: \code{"Size of dimension
 #dimno of argument '#arg' must be #dimcomp#dim_req, but is actually
 #dim_is."})

\item Wrong number of dimensions of parameter (default: \code{"Number of
 dimensions of argument '#arg' must be #dimcnt_req but is actually
 #dimcnt_is."})

\item Parameter value is not permitted (default: \code{"#argval is not a
 valid value for argument #arg."})

}

The messages provided via the \code{messages} argument are templates that
can use predefined placeholders to convey information relevant for
understanding the problem.
}

\subsection{Placeholder that can be used in message templates}{

\itemize{

\item \emph{#fun}: The name of the function of which the parameter values
are to be checked (i.e. the function \code{check_types()} is applied to)

\item \emph{#arg}: The name of the argument

\item \emph{#argval}: The value of the parameter used in the function call

\item \emph{#type_req}: The required type for the argument

\item \emph{#type_is}: The actual type of the parameter used in the function
call

\item \emph{#dimcnt_req}: The required number of dimensions of the argument

\item \emph{#dimcnt_is}: The actual number of dimensions of the parameter
used in the function call

\item \emph{#dim_req}: The required size of the dimension where a dimension
size error occurred

\item \emph{#dim_is}: The actual size of the dimension where a dimension
size error occurred

\item \emph{#dimcomp}: The comparison operator used in combination with
\code{#dim_req}, the required size of the dimension (e.g. the \code{>=} in
\code{>=2}, if this dimension of the argument is to be greater than 1)

\item \emph{#dimno}: The index of the dimension where a dimension size error
occurred

}
}
}

\examples{

celsius_to_fahrenheit <- function(degrees_celsius) {
 #| degrees_celsius numeric dim(1) not(NA, NULL)

 if(check_types()) return(degrees_celsius * 9/5 + 32)
 else return(NA)

}
}
\seealso{
Other typehint: 
\code{\link{show_typehints}()},
\code{\link{typehint}}
}
\concept{typehint}
