% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/polish.uniLasso.R
\name{polish.uniLasso}
\alias{polish.uniLasso}
\title{Fit a cross-validated univariate guided lasso model, followed by a lasso polish.}
\usage{
polish.uniLasso(
  x,
  y,
  family = c("gaussian", "binomial", "cox"),
  weights = NULL,
  ...
)
}
\arguments{
\item{x}{Input matrix, of dimension \code{nobs x nvars}; each row is
an observation vector.}

\item{y}{Response variable. Quantitative for \code{family = "gaussian"} or
\code{family = "poisson"} (non-negative counts). For \code{family="binomial"},
should be a numeric vector consisting of 0s and 1s. For \code{family="cox"},
y should be a two-column matrix with columns named 'time' and 'status'.
The latter is a binary variable, with '1' indicating death, and '0'
indicating right-censored.}

\item{family}{one of "gaussian","binomial" or "cox". Currently only these families are implemented.
In the future others will be added.}

\item{weights}{optional vector of non-negative weights, default is NULL which results in all weights = 1.}

\item{\ldots}{additional arguments passed to \code{cv.uniLasso} and
\code{cv.glmnet}. Note: by defaults \code{cv.uniLasso()} uses
\code{standardize=FALSE}, and \code{cv.glmnet()} uses
\code{standardize=TRUE}. These are both the sensible defaults for
this function. Users can supply \code{standardize=FALSE} (via the
\ldots argument) which will overide the \code{cv.glmnet()}
default. Users should avoid using \code{standardize=TRUE}, since
this will affect the first stage model as well, where this is
not a suitable choice.}
}
\value{
An object of class \code{"polish.unilasso"} that inherits
from class \code{"cv.glmnet"}. The \code{"glmnet.fit"} is the
\emph{stitched} second-stage model, from which predictions are
made. An additional component named \code{"cv.uniLasso"} is the
first stage model.
}
\description{
This function has two stages. In the first we fit a
univariate-guided sparse regression \code{uniLasso} model using
cross-validation to select the lasso penalty
parameter (using \code{s = "lambda.min"}). In the second stage, we use the predictions from this chosen
model as an offset, and fit a cross-validated unrestricted lasso
model. For squared error loss, this means we post-fit a lasso model
to the residuals. Conveniently, it returns an object that inherits
from \code{cv.glmnet}, in which the two models are \emph{stitched}
together. What this means is that the chosen coefficients from the
first model are added to the coefficients from the second, and other related components are updated as well.
This means at predict time we do not have to fiddle with offsets.  All
of the methods for \code{cv.glmnet} can be applied, such as
\code{predict}, \code{plot}, \code{coef}, \code{print}, and \code{assess.glmnet}.
}
\examples{

# Gaussian data, p=1000, n=300, SNR=1  "medium SNR"
# use the built-in simulate function to create Gaussian data
set.seed(101)
data <- simulate_uniLasso("medium-SNR")
attach(data) # has components "x","y","xtest","ytest","mutest","sigma"
pfit <- polish.uniLasso(x,y)
plot(pfit)
pred <- predict(pfit, newx = xtest,  s = "lambda.min") # ie predict from a "cv.glmnet" object.
mean((ytest-pred)^2) # test error
print(pfit)
print(pfit$glmnet.fit)
plot(pfit$glmnet.fit) # coefficient plot of the second stage
plot(pfit$cv.uniLasso) # cv.glmnet plot of the first stage
plot(pfit$cv.uniLasso$glmnet.fit) # coefficient plot of the first stage


# Binomial response

yb =as.numeric(y>0)
pfitb = polish.uniLasso(x, yb, family="binomial")
predict(pfitb, xtest[1:10,], type="response") # predict at default s = "lambda.1se"
plot(pfitb)
plot(pfitb$glmnet.fit) # plot second stage lasso coefficient path
plot(pfitb$cv.uniLasso) # plot first stage cv.uniLasso results
# Cox response

set.seed(10101)
N = 1000
p = 30
nzc = p/3
x = matrix(rnorm(N * p), N, p)
beta = rnorm(nzc)
fx = x[, seq(nzc)] \%*\% beta/3
hx = exp(fx)
ty = rexp(N, hx)
tcens = rbinom(n = N, prob = 0.3, size = 1)  # censoring indicator
y = cbind(time = ty, status = 1 - tcens)  # y=Surv(ty,1-tcens) with library(survival)
pfitc = polish.uniLasso(x, y, family = "cox")
plot(pfitc)
plot(pfitc$cv.uniLasso)

}
