% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate.r
\name{dfsim}
\alias{dfsim}
\title{Generalized Dose-Finding Ensemble Simulator}
\usage{
dfsim(
  n,
  starting = NULL,
  sprobs = NULL,
  cohort = 1,
  Fvals,
  ensemble = dim(Fvals)[2],
  design = krow,
  desArgs = list(k = 1),
  thresholds = NULL,
  seed = NULL,
  showdots = TRUE
)
}
\arguments{
\item{n}{sample size}

\item{starting}{the starting dose level. If \code{NULL} (default), will be randomized.}

\item{sprobs}{the probability weights if using a randomized starting dose. If \code{NULL} (default) will be discrete-uniform.}

\item{cohort}{the cohort (group) size, default 1.}

\item{Fvals}{(vector or matrix): the true values of the response function on the dose grid. These are the dose-response scenarios from which the experimental runs will be simulated. If running an ensemble with different scenarios, each scenarios is a column. If running an identical-scenario ensemble, provide a single vector as well as \code{ensemble}.}

\item{ensemble}{the number of different runs/scenarios to be simulated. Will be determined automatically if \code{Fvals} is a matrix, as the number of columns.}

\item{design}{the dose-finding design function used to determine the next dose. Default \code{krow}; see \code{\link{krow}} for options.}

\item{desArgs}{List of arguments passed on to \code{design}. Need to be compatible for use in \code{mapply}. Default is \code{list(k=1)}, which together with \code{design = krow} will generate a Clasical (median-finding) UDD simulation.}

\item{thresholds}{Matrix of size (at least) \code{n} by \code{ensemble}, the response thresholds of participants, presented as percentiles (i.e., output of \code{runif()}) rather than physical values. If \code{NULL} (default), they will be simulated on the fly. When running comparative performance simulations, we recommend providing the same thresholds to everything you want to compare on equal footing.}

\item{seed}{The random seed if simulating the thresholds. Can be kept \emph{"floating"} (i.e., varying between calls) if left as \code{NULL} (default).}

\item{showdots}{Logical: print out a dot (\code{.}) after each designion step in \code{1:n}, and the start/end time stamps? Default \code{TRUE}.}
}
\value{
A list with the following elements:
\itemize{
\item \code{scenarios}: \code{Fvals}
\item \code{sample}: \code{thresholds}
\item \code{doses}: The matrix of simulated dose allocations for each run (\code{n+1} by \code{ensemble})
\item \code{responses}: The matrix of simulated responses (0 or 1) for each run (\code{n} by \code{ensemble})
\item \code{cohort}: \code{cohort}
\item \code{details}: \code{desArgs}
}
}
\description{
This function simulates sequential dose-finding experiments on a fixed dose grid. The response function is (implicitly) assumed monotone in \verb{(0,1)}
}
\details{
A vectorized dose-finding simulator, set up to run an entire ensemble simultaneously.
The simulated doses are indices \code{1:nlev} with \code{nlev} being the number of dose levels.
Upon output they can be optionally "dressed up" with physical values using the \code{xvals} argument.

The simulator's essential use within the \code{upndown} package is to estimate bootstrap confidence intervals for dose-averaging target estimates, via \code{\link{dfboot}}. But it can be also used stand-alone as a study-design aid.

The particular dose-finding design simulated is determined by \code{design} and its argument list \code{desArgs}. The 3 straightforward extensions of the median-finding "Classical" UDD are available, namely "k-in-a-row", biased-coin and group (cohort) UDD. To simulate the the median-finding "Classical" UDD itself, use \code{krow} with \code{desArgs = list(k=1)}.
Other non-UDD dose-finding designs - e.g., CRM, CCD, BOIN, etc. - can also be made compatible with \code{dfsim}. Utilities to run those 3 in particular are available on GitHub, under \code{assaforon/UpndownBook/P3_Practical/OtherDesigns.r}.

If you want to create a \code{design} function yourself, it would need to accept \verb{doses, responses} as input, and return the next dose allocation (as an integer index).
The main progression loop is run via \code{mapply}.
}
\note{
This is an adaptation of a non-package function used by the author for well over a decade before incorporating it into \code{upndown} in late 2023. For initial guidance, see the code example. If you encounter any funny behavior please let me know. Thank you!
}
\examples{
### dfsim example

# We provide a "toy example" for how randomized-F simulations might work.
# We have been strong advocated of this simulation approach, especially for performance comparison
#   between designs and between estimators. It is far preferable to the "cherry-picked F" approach.
# Unfortunately, the latter is still more commonly found in dose-finding literature.

# At core is a randomized ensemble of F(x) ("dose-response") curves. It is far easier
#   to generate parametric ensembles rather than nonparametric or semi-parametric ones.
# So this is what we do here. We use the Weibull, being capable of the most diverse ensembles
#   among common parametric families.

# This being a toy example, it is more simplistic than our current practice. For the latter,
#   see either the supplement to Oron et al. 2022, or the NE Journal of SDS due online
#   late 2024 or early 2025.  Ok, let's go!

# We simulate 7-level experiments.
m = 7
# And 10 curves in total
B = 10
# Parameter generation (I chose the parameter bounds after some trial and error)
# Note I am not fixing a seed, so each time you run this you'll get a different ensemble!
wparams = cbind(2^runif(B, -2, 2.5), runif(B, 1, 10) )
round(wparams, 2)
# Now the F(x) curves; they should be in columns
ensemble = apply(wparams, 1, function(x, doses = m) 
                  pweibull(1:doses, shape = x[1], scale = x[2]) )

# Let's see what we got!
round(ensemble, 3)
# The experiment will be "Classical" median-targeting. In real life if 0.5 falls outside
#     of the range of doses, it's not great. For simulation it's fine; it'll enable us to 
#     watch the allocations for such curves heap near the edge with F closest to 0.5.

# Let the experiments be a measly n=15, to keep runtime under the menacing 5 seconds.
# We start all runs smack in the middle, level 4:
sout = dfsim(n = 15, starting = 4, Fvals = ensemble, design = krow, desArgs = list(k=1) )
names(sout)

# "scenarios" are the F values we provided, while "sample" is the set of randomized thresholds
#   simulated from the uniform distribution. If you run comparisons, we recommend that
#   you only randomize the first set in the comparison, and feed the very same thresholds to 
#   all the rest.

# Anyhoo, here are the simulated trajectories. 
#     Note that there are n+1 x values in each one, because after 
#     seeing x_n and y_n, the n+1-th allocation can be determined.
sout$doses
# Compare with the F ensemble. Is it what you expected? 
# Probably more random-walky than you thought :)

# The binary responses are below. Before going big on the simulation, it's a good idea to
#    sanity-check and see that the doses and responses match according to the design's rules.
sout$responses
# Some meta details about the design and simulation settings are available here:
sout$details


}
\seealso{
\itemize{
\item \code{\link{dfboot}}
}
}
\author{
Assaf P. Oron
}
