% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/critical_points_nd.R
\name{critical_points_nd}
\alias{critical_points_nd}
\title{Critical points of a scalar field in n dimensions (no plot)}
\usage{
critical_points_nd(
  f,
  bounds,
  start_grid = NULL,
  n_random = 50L,
  max_grid_starts = 2000L,
  h = NULL,
  tol_grad = 1e-06,
  tol_merge = 0.001,
  tol_eig = 1e-06,
  maxit = 200,
  optim_method = c("BFGS", "Nelder-Mead"),
  seed = NULL,
  store_hessian = FALSE
)
}
\arguments{
\item{f}{Scalar field as \code{function(x)}, where \code{x} is a numeric
vector of length \code{n}, returning a numeric scalar.}

\item{bounds}{Domain bounds. Either:
\itemize{
\item an \code{n x 2} numeric matrix or data frame with columns
\code{lower} and \code{upper}, or
\item a list of length \code{n}, each element a numeric vector
\code{c(lower, upper)}.
}}

\item{start_grid}{Integer vector of length \code{n} with the number of
regular grid points per dimension used as deterministic starting points.
If \code{NULL}, a default \code{rep(5, n)} is used. The total number of
grid points is limited by \code{max_grid_starts}.}

\item{n_random}{Integer. Number of additional random starting points
sampled uniformly inside the domain defined by \code{bounds}.}

\item{max_grid_starts}{Maximum number of deterministic grid starting
points that are actually used. If the full grid would exceed this
value, a random subset of that size is taken.}

\item{h}{Step size for finite differences. Can be:
\itemize{
\item \code{NULL}: automatic componentwise step size
\code{1e-4 * (1 + abs(x_i))},
\item a numeric scalar: same step for all coordinates,
\item a numeric vector of length \code{n}: one step per coordinate.
}}

\item{tol_grad}{Numeric threshold on the gradient norm used to accept
a point as critical. Smaller values make the criterion more strict.}

\item{tol_merge}{Numeric radius used to merge nearby critical point
candidates (Euclidean distance).}

\item{tol_eig}{Numeric tolerance used to decide whether Hessian
eigenvalues are treated as positive, negative or close to zero
for classification.}

\item{maxit}{Maximum number of iterations allowed for each call to
\code{stats::optim()}.}

\item{optim_method}{Primary optimization method passed to
\code{stats::optim()}, for example \code{"BFGS"} or \code{"Nelder-Mead"}.
If the primary method fails with an error, the function falls back
to \code{"Nelder-Mead"}.}

\item{seed}{Optional integer seed for reproducibility of the random
starting points.}

\item{store_hessian}{Logical. If \code{TRUE}, the Hessian matrix at each
critical point is stored and returned.}
}
\value{
A list with components:
\describe{
\item{\code{critical_points}}{
A data frame with columns \code{x1, ..., xn}, the function value
\code{f}, the gradient norm \code{grad_norm}, and the classification
label \code{class}.
}
\item{\code{eigvals}}{
A list of numeric vectors containing the Hessian eigenvalues for each
critical point.
}
\item{\code{hessians}}{
If \code{store_hessian = TRUE}, a list of Hessian matrices
(one per critical point). Otherwise \code{NULL}.
}
\item{\code{starts_info}}{
A list with information about the number of grid and random starting
points actually used.
}
}
}
\description{
Searches for approximate critical points of a scalar field
\code{f(x)} in dimension \code{n >= 3} over a rectangular domain.
The algorithm looks for points where the gradient is close to zero by
minimizing the squared gradient norm \code{g(x) = ||grad f(x)||^2}
from multiple starting points.
}
\details{
Candidate points that are closer than \code{tol_merge} (in Euclidean
distance) are merged into a single representative. Each accepted point
is classified by the eigenvalues of the numerical Hessian as
\code{"minimum"}, \code{"maximum"}, \code{"saddle"} or \code{"flat"}.

Gradients and Hessians are computed with second-order central finite
differences.
}
\examples{
\donttest{
# Example 1: unique minimum at (1, 1, 1)
f1 <- function(x) sum((x - 1)^2)
B  <- rbind(c(-2, 3), c(-2, 3), c(-2, 3))  # 3D bounds
res1 <- critical_points_nd(
  f1,
  bounds      = B,
  start_grid  = c(5, 5, 5),
  n_random    = 50,
  seed        = 1
)
res1$critical_points

# Example 2: saddle at the origin in 3D
f2 <- function(x) x[1]^2 + x[2]^2 - x[3]^2
B2 <- rbind(c(-1, 1), c(-1, 1), c(-1, 1))
res2 <- critical_points_nd(
  f2,
  bounds      = B2,
  start_grid  = c(5, 5, 5),
  n_random    = 30,
  seed        = 123
)
res2$critical_points

# Example 3 (4D): multiple critical points
f3 <- function(x) sum(x^4 - 2 * x^2)
B3 <- do.call(rbind, replicate(4, c(-2, 2), simplify = FALSE))
res3 <- critical_points_nd(
  f3,
  bounds      = B3,
  start_grid  = rep(4, 4),
  n_random    = 200,
  seed        = 42
)
head(res3$critical_points)
}

}
