% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/curl3d.R
\name{curl3d}
\alias{curl3d}
\title{Numerical curl of a three-dimensional vector field}
\usage{
curl3d(field, x, y, z, h = NULL, tval = 0, method = c("central"))
}
\arguments{
\item{field}{A function representing the vector field. It can be defined
as \code{function(x, y, z)} or as \code{function(x, y, z, t)}.
It must return a numeric vector \code{c(Fx, Fy, Fz)}.}

\item{x, y, z}{Numeric scalars giving the coordinates of the evaluation
point.}

\item{h}{Step size for finite differences. It may be:
\itemize{
\item a single numeric value,
\item a numeric vector of length three,
\item or \code{NULL} (automatic selection).
}}

\item{tval}{Time value used when the vector field depends on time.
Default is \code{0}.}

\item{method}{Differencing scheme. Currently only \code{"central"} is
supported.}
}
\value{
A named numeric vector of length three containing the curl
components at the evaluation point. The components are named
\code{omega_x}, \code{omega_y} and \code{omega_z}.
}
\description{
Computes the curl of a vector field in three dimensions at a given
point, using second-order central finite differences.
The field may optionally depend on a time parameter; if so, the curl
is evaluated at a fixed time value.
}
\details{
The vector field must be a function that returns a numeric vector of
length three representing the components of the field at a point.
The curl is obtained by approximating the partial derivatives of the
field components with respect to each coordinate direction using
symmetric finite differences.

The step size for each coordinate can be:
\itemize{
\item a single scalar used for all three axes,
\item a numeric vector of length three providing separate steps for
the x, y and z directions,
\item or \code{NULL}, in which case automatic step sizes are chosen
based on the evaluation point.
}

The method currently implemented is the second-order central
differencing scheme. Smaller step sizes may provide more accurate
results for rapidly varying fields, at the cost of increased
sensitivity to floating-point error.
}
\examples{
# Simple rotating field: curl is constant in the third component
field1 <- function(x, y, z) c(-y, x, 0.6)
curl3d(field1, x = 0.1, y = -0.3, z = 2)

# Time-dependent example (time does not affect the curl):
field2 <- function(x, y, z, t) c(-y, x + t, z)
curl3d(field2, x = 1, y = 2, z = 3, tval = 5)

# Using a smaller step size for more precision:
curl3d(field1, x = 1, y = 1, z = 1, h = 1e-5)

}
