% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vmideal_vst.R
\name{vmidealVst}
\alias{vmidealVst}
\alias{vmidealVstFromMagn}
\alias{vmidealVstToPsi}
\title{Variance-stabilizing Transformation for the Ideal Distribution of Visual Meteor Magnitudes}
\usage{
vmidealVstFromMagn(m, lm)

vmidealVstToPsi(tm, lm, deriv.degree = 0L)
}
\arguments{
\item{m}{integer; the meteor magnitude.}

\item{lm}{numeric; limiting magnitude.}

\item{tm}{numeric; transformed magnitude.}

\item{deriv.degree}{integer; the degree of the derivative at \code{tm} to return
instead of \code{psi}. Must be \code{0}, \code{1} or \code{2}.}
}
\value{
\itemize{
\item \code{vmidealVstFromMagn}: a numeric value, the transformed meteor magnitude.
\item \code{vmidealVstToPsi}: a numeric value of the parameter \code{psi}, derived from
the mean of \code{tm}.
The argument \code{deriv.degree} can be used to apply the delta method.
}
}
\description{
Applies a variance-stabilizing transformation to meteor magnitudes
under the assumption of the ideal magnitude distribution.
}
\details{
Many linear models require the variance of visual meteor magnitudes to be
homoscedastic. The function \code{vmidealVstFromMagn} applies a transformation
that produces homoscedastic distributions of visual meteor magnitudes if the
underlying magnitudes follow the ideal magnitude distribution.
In this sense, the transformation acts as a normalization of meteor magnitudes
and yields a variance close to \code{1.0}.

The ideal distribution of visual meteor magnitudes
depends on the \link[=vmideal]{parameter psi} and the limiting magnitude \code{lm},
resulting in a two-parameter distribution. Without detection probabilities,
the magnitude distribution reduces to a pure \link[=mideal]{ideal magnitude distribution},
which depends only on the parameter \code{psi}. Since the
limiting magnitude \code{lm} is a fixed parameter and never estimated
statistically, the magnitudes can be transformed such that, for example,
the mean of the transformed magnitudes directly provides an estimate of \code{psi}
using the function \code{vmidealVstToPsi}.

This transformation is valid for \eqn{-10 \le \texttt{psi} \le 9}.
The numerical form of the transformation is version-specific and may change
substantially in future releases. Do not rely on equality of transformed
values across package versions.
}
\note{
The internal approximations used here are derived from the perception
probabilities produced by \link{vmperception}.
For details on the derivation, see the script \code{inst/derivation/vmideal_vst.R} in the
package's source code.
}
\examples{
N <- 100
psi <- 5.0
limmag <- 6.3

# Simulate magnitudes
m <- rvmideal(N, limmag, psi)

# Variance-stabilizing transformation
tm <- vmidealVstFromMagn(m, limmag)
tm.mean <- mean(tm)
tm.var  <- var(tm)

# Estimator for psi from the transformed mean
psi.hat  <- vmidealVstToPsi(tm.mean, limmag)

# Derivative d(psi)/d(tm) at tm.mean (needed for the delta method)
dpsi_dtm <- vmidealVstToPsi(tm.mean, limmag, deriv.degree = 1L)

# Variance of the sample mean of tm
var_tm.mean <- tm.var / N

# Delta method: variance and standard error of psi.hat
var_psi.hat <- (dpsi_dtm^2) * var_tm.mean
se_psi.hat  <- sqrt(var_psi.hat)

# Results
print(psi.hat)
print(se_psi.hat)
}
\seealso{
\link{vmideal} \link{mideal} \link{vmperception}
}
