% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcs_delay.R
\name{mcs_delay.default}
\alias{mcs_delay.default}
\title{Adjustment of Operating Times by Delays using a Monte Carlo Approach}
\usage{
\method{mcs_delay}{default}(
  ...,
  date_1,
  date_2,
  time,
  status = NULL,
  id = paste0("ID", seq_len(length(time))),
  distribution = c("lognormal", "exponential")
)
}
\arguments{
\item{...}{Further arguments passed to or from other methods. Currently not used.}

\item{date_1}{A vector of class \code{character} or \code{Date}, in the format "yyyy-mm-dd",
representing the earlier of the two dates belonging to a particular delay.
Use \code{NA} for missing elements.

If more than one delay is to be considered, use a list where the first element
is the earlier date of the first delay, the second element is the earlier date
of the second delay, and so forth (see 'Examples').}

\item{date_2}{A vector of class \code{character} or \code{Date} in the format "yyyy-mm-dd".
\code{date_2} is the counterpart of \code{date_1} and is used the same as \code{date_1}, just with
the later date(s) of the particular delay(s). Use \code{NA} for missing elements.}

\item{time}{Operating times. Use \code{NA} for missing elements.}

\item{status}{Optional argument. If used, it must contain binary data
(0 or 1) indicating whether a unit is a right censored observation (= 0) or a
failure (= 1).

If \code{status} is provided, class \code{wt_reliability_data} is assigned to the
output of \link{mcs_delay}, which enables the direct application of \link{estimate_cdf}
on operating times.}

\item{id}{Identification of every unit.}

\item{distribution}{Supposed distribution of the respective delay.}
}
\value{
A list with class \code{wt_mcs_delay} containing the following elements:
\itemize{
\item \code{data} : A \code{tibble} returned by \link{mcs_delay_data} where two modifications
has been made:
\itemize{
\item If the column \code{status} exists, the \code{tibble} has additional classes
\code{wt_mcs_data} and \code{wt_reliability_data}. Otherwise, the \code{tibble} only has
the additional class \code{wt_mcs_data} (which is not supported by \link{estimate_cdf}).
\item The column \code{time} is renamed to \code{x} (to be in accordance with
\link{reliability_data}) and contains the adjusted operating times for incomplete
observations and input operating times for the complete observations.
}
\item \code{sim_data} : A \code{tibble} with column \code{sim_delay} that holds the simulated
delay-specific numbers for incomplete cases and \code{0} for complete cases.
If more than one delay was considered multiple columns with names \code{sim_delay_1},
\code{sim_delay_2}, ..., \code{sim_delay_i} and corresponding delay-specific random
numbers are presented.
\item \code{model_estimation} : A list returned by \link{dist_delay}.
}
}
\description{
In general, the amount of available information about units in the field is very
different. During the warranty period, there are only a few cases with complete
data (mainly \emph{failed units}) but lots of cases with incomplete data (usually
\emph{censored units}). As a result, the operating time of units with incomplete
information is often inaccurate and must be adjusted by delays.

This function reduces the operating times of incomplete observations by simulated
delays (in days). A unit is considered as incomplete if the later of the
related dates is unknown. See 'Details' for some practical examples.

Random delay numbers are drawn from the distribution determined by complete cases
(described in 'Details' of \link{dist_delay}).
}
\details{
In field data analysis time-dependent characteristics (e.g. \emph{time in service})
are often imprecisely recorded. These inaccuracies are caused by unconsidered delays.

For a better understanding of the MCS application in the context of field data,
two cases are described below.
\itemize{
\item \strong{Delay in registration}: It is common that a supplier, which provides
parts to the manufacturing industry does not know when the unit, in which
its parts are installed, were put in service (due to unknown registration or
sales date (\code{date_2})). Without taking the described delay into account, the
time in service of the failed units would be the difference between the
repair date and the production date (\code{date_1}) and for intact units the
difference between the present date and the production date. But the real
operating times are (much) shorter, since the stress on the components have
not started until the whole systems were put in service. Hence, units with
incomplete data (missing \code{date_2}) must be reduced by the delays.
\item \strong{Delay in report}:: Authorized repairers often do not immediately
notify the manufacturer or OEM of repairs that were made during the warranty
period, but instead pass the information about these repairs in collected
forms e.g. weekly, monthly or quarterly. The resulting time difference between
the reporting (\code{date_2}) of the repair in the guarantee database and the
actual repair date (\code{date_1}), which is often assumed to be the failure
date, is called the reporting delay. For a given date where the analysis
is made there could be units which had a failure but the failure isn't
reported and therefore they are treated as censored units. In order to take
this into account and according to the principle of equal opportunities, the
lifetime of units with missing report date (\code{date_2[i] = NA}) is reduced by
simulated reporting delays.
}
}
\examples{
# Example 1 - MCS for delay in registration:
mcs_regist <- mcs_delay(
  date_1 = field_data$production_date,
  date_2 = field_data$registration_date,
  time = field_data$dis,
  status = field_data$status,
  distribution = "lognormal"
)

# Example 2 - MCS for delay in report:
mcs_report <- mcs_delay(
  date_1 = field_data$repair_date,
  date_2 = field_data$report_date,
  time = field_data$dis,
  status = field_data$status,
  distribution = "exponential"
)

# Example 3 - Reproducibility of random numbers:
set.seed(1234)
mcs_report_reproduce <- mcs_delay(
  date_1 = field_data$repair_date,
  date_2 = field_data$report_date,
  time = field_data$dis,
  status = field_data$status,
  distribution = "exponential"
)

# Example 4 - MCS for delays in registration and report with same distribution:
mcs_delays <- mcs_delay(
  date_1 = list(field_data$production_date, field_data$repair_date),
  date_2 = list(field_data$registration_date, field_data$report_date),
  time = field_data$dis,
  status = field_data$status,
  distribution = "lognormal"
)

# Example 5 - MCS for delays in registration and report with different distributions:
## Assuming lognormal registration and exponential reporting delays.
mcs_delays_2 <- mcs_delay(
  date_1 = list(field_data$production_date, field_data$repair_date),
  date_2 = list(field_data$registration_date, field_data$report_date),
  time = field_data$dis,
  status = field_data$status,
  distribution = c("lognormal", "exponential")
)

}
\references{
Verband der Automobilindustrie e.V. (VDA); Qualitätsmanagement in
der Automobilindustrie. Zuverlässigkeitssicherung bei Automobilherstellern
und Lieferanten. Zuverlässigkeits-Methoden und -Hilfsmittel.; 4th Edition, 2016,
ISSN:0943-9412
}
\seealso{
\link{dist_delay} for the determination of a parametric delay distribution
and \link{estimate_cdf} for the estimation of failure probabilities.
}
