% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/org_fams.R
\name{org_fams}
\alias{org_fams}
\title{Organize animals into families and expand pedigree data}
\usage{
org_fams(ped, sampledata, output = "both")
}
\arguments{
\item{ped}{Data frame. \code{FamAgg} output of \code{\link[=get_colony]{get_colony()}} or \code{\link[=get_ped]{get_ped()}} function.
With \code{rm_obsolete_parents} parameter set to \code{TRUE}.}

\item{sampledata}{Data frame. Metadata for all genetic samples that belong
to the individuals included in pedigree reconstruction analysis.
This data frame should adhere to the formatting and naming conventions
outlined in the \code{\link[=check_sampledata]{check_sampledata()}} documentation.}

\item{output}{Character string. Determines the format of the output. Options are:
"ped": returns an extended pedigree data frame.
"fams": returns a table of all families present in the pedigree.
"both": returns a list with two data frames: "ped" and "fams". (Default)}
}
\value{
Depending on the \code{output} parameter, the function returns either a data frame
(\code{ped} or \code{fams}) or a list containing both data frames (\code{ped} and \code{fams}).
\itemize{
\item \code{ped} data frame. An extended version of the pedigree data from \code{get_colony()}/\code{get_ped()}.
In addition to common pedigree information (individual, mother, father, sex,
family), \code{ped} includes columns for:
\itemize{
\item \code{parents}: Identifier codes of both parents separated with \verb{_}.
\item \code{FamID}: Numeric identifier for the family to which the individual belongs (see \code{fams} below).
\item \code{FirstSeen}: Date of first sample of individual.
\item \code{LastSeen}: Date of last sample of individual.
\item \code{IsDead}: Logical value (\code{TRUE/FALSE}) that identifies if the individual is dead.
\item \code{DadHSgroup}: Identifier of paternal half-sib group (see Details).
\item \code{MomHSgroup}: Identifier of maternal half-sib group (see Details).
\item \code{hsGroup}: Numeric value indicating if the individual is part of
a half-sib group (see Details).
}
\item \code{fams} data frame includes information on families that individuals in the pedigree
belong to. The families are described by:
\itemize{
\item \code{parents}: Identifier codes of both parents separated with \verb{_}.
\item \code{father}: Identifier code of the father.
\item \code{mother}: Identifier code of the mother.
\item \code{FamID}: Numeric identifier for the family.
\item \code{famStart}: Date when the first sample of one of the offspring from this family was collected (see Details).
\item \code{famEnd}: Date when the last sample of mother or father of this family was collected (see Details).
\item \code{FamDead}: Logical value (\code{TRUE/FALSE}) indicating if the family no longer exists.
\item \code{DadHSgroup}: Identifier connecting families that share the same father.
\item \code{MomHSgroup}: Identifier connecting families that share the same mother.
\item \code{hsGroup}: Numeric value connecting families that share one of the parents.
}
}
}
\description{
Takes pedigree data from \code{\link[=get_colony]{get_colony()}} or \code{\link[=get_ped]{get_ped()}} function and groups animals into families.
It also expands the pedigree data by adding information about the family that each individual was born in and the
family in which the individual is the reproductive animal.
}
\details{
The result of \code{org_fams()} function introduces us to two important concepts
within the context of this package: family and half-sib group. A family in the
output of this function is defined as a group of animals where at least one
parent and at least one offspring is known. A half-sib group refers to a
group of half-siblings, either maternally or paternally related. In the
function output the \code{DadHSgroup} groups paternal half-siblings and \code{MomHSgroup}
maternal half-siblings.

The \code{fams} output dataframe contains \code{famStart} and \code{famEnd} columns, which estimate
a time window for the family based solely on sample collection dates provided in \code{sampledata}.
\code{famStart} marks the date of the earliest sample collected from any offspring
belonging to that family. \code{famEnd} indicates the date of the latest sample collected
from either the mother or the father of that family. It is important to recognize that this
method relies on observation (sampling) times. Consequently, \code{famEnd} (last parental sample date)
can precede \code{famStart} (first offspring sample date), creating a biologically impossible sequence
and a negative calculated family timespan. Users should interpret the interval
between \code{famStart} and \code{famEnd} with this understanding.
}
\examples{

# Prepare the data for usage with org_fams() function.
# Get animal timespan data using the anim_timespan() function.
animal_ts <- anim_timespan(
  wolf_samples$AnimalRef,
  wolf_samples$Date,
  wolf_samples$SType,
  dead = c("Tissue")
)
# Add animal timespan to the sampledata
sampledata <- merge(wolf_samples, animal_ts, by.x = "AnimalRef", by.y = "ID", all.x = TRUE)
# Define the path to the pedigree data file.
path <- paste0(system.file("extdata", package = "wpeR"), "/wpeR_samplePed")
# Retrieve the pedigree data from the get_colony function.
ped_colony <- get_colony(path, sampledata, rm_obsolete_parents = TRUE, out = "FamAgg")

# Run the function
# Organize families and expand pedigree data using the org_fams function.
org_fams(
    ped = ped_colony,
    sampledata = sampledata
    )


}
