use crate::iter_set::{Iter, OwningIter};
#[cfg(feature = "raw-api")]
use crate::lock::RwLock;
use crate::setref::one::Ref;
use crate::DashMap;
#[cfg(feature = "raw-api")]
use crate::HashMap;
use cfg_if::cfg_if;
use core::borrow::Borrow;
use core::fmt;
use core::hash::{BuildHasher, Hash};
use core::iter::FromIterator;
use std::collections::hash_map::RandomState;





pub struct DashSet<K, S = RandomState> {
    pub(crate) inner: DashMap<K, (), S>,
}

impl<K: Eq + Hash + fmt::Debug, S: BuildHasher + Clone> fmt::Debug for DashSet<K, S> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&self.inner, f)
    }
}

impl<K: Eq + Hash + Clone, S: Clone> Clone for DashSet<K, S> {
    fn clone(&self) -> Self {
        Self {
            inner: self.inner.clone(),
        }
    }

    fn clone_from(&mut self, source: &Self) {
        self.inner.clone_from(&source.inner)
    }
}

impl<K, S> Default for DashSet<K, S>
where
    K: Eq + Hash,
    S: Default + BuildHasher + Clone,
{
    fn default() -> Self {
        Self::with_hasher(Default::default())
    }
}

impl<'a, K: 'a + Eq + Hash> DashSet<K, RandomState> {
    
    
    /// # Examples
    
    
    
    
    
    
    
    pub fn new() -> Self {
        Self::with_hasher(RandomState::default())
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    
    pub fn with_capacity(capacity: usize) -> Self {
        Self::with_capacity_and_hasher(capacity, RandomState::default())
    }
}

impl<'a, K: 'a + Eq + Hash, S: BuildHasher + Clone> DashSet<K, S> {
    
    
    /// # Examples
    
    
    
    
    
    
    
    
    
    pub fn with_hasher(hasher: S) -> Self {
        Self::with_capacity_and_hasher(0, hasher)
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    
    
    
    pub fn with_capacity_and_hasher(capacity: usize, hasher: S) -> Self {
        Self {
            inner: DashMap::with_capacity_and_hasher(capacity, hasher),
        }
    }

    
    
    pub fn hash_usize<T: Hash>(&self, item: &T) -> usize {
        self.inner.hash_usize(item)
    }

    cfg_if! {
        if #[cfg(feature = "raw-api")] {
            
            
            
            
            
            /// # Examples
            
            
            
            
            
            
            
            pub fn shards(&self) -> &[RwLock<HashMap<K, (), S>>] {
                self.inner.shards()
            }
        }
    }

    cfg_if! {
        if #[cfg(feature = "raw-api")] {
            
            
            
            
            
            
            /// # Examples
            
            
            
            
            
            
            
            
            pub fn determine_map<Q>(&self, key: &Q) -> usize
            where
                K: Borrow<Q>,
                Q: Hash + Eq + ?Sized,
            {
                self.inner.determine_map(key)
            }
        }
    }

    cfg_if! {
        if #[cfg(feature = "raw-api")] {
            
            
            
            
            /// # Examples
            
            
            
            
            
            
            
            
            
            pub fn determine_shard(&self, hash: usize) -> usize {
                self.inner.determine_shard(hash)
            }
        }
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    pub fn insert(&self, key: K) -> bool {
        self.inner.insert(key, ()).is_none()
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    
    pub fn remove<Q>(&self, key: &Q) -> Option<K>
    where
        K: Borrow<Q>,
        Q: Hash + Eq + ?Sized,
    {
        self.inner.remove(key).map(|(k, _)| k)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn remove_if<Q>(&self, key: &Q, f: impl FnOnce(&K) -> bool) -> Option<K>
    where
        K: Borrow<Q>,
        Q: Hash + Eq + ?Sized,
    {
        
        self.inner.remove_if(key, |k, _| f(k)).map(|(k, _)| k)
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    
    pub fn iter(&'a self) -> Iter<'a, K, S, DashMap<K, (), S>> {
        let iter = self.inner.iter();

        Iter::new(iter)
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    
    pub fn get<Q>(&'a self, key: &Q) -> Option<Ref<'a, K, S>>
    where
        K: Borrow<Q>,
        Q: Hash + Eq + ?Sized,
    {
        self.inner.get(key).map(Ref::new)
    }

    
    pub fn shrink_to_fit(&self) {
        self.inner.shrink_to_fit()
    }

    
    
    
    /// # Examples
    
    
    
    
    
    
    
    
    
    
    
    pub fn retain(&self, mut f: impl FnMut(&K) -> bool) {
        self.inner.retain(|k, _| f(k))
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    
    
    
    pub fn len(&self) -> usize {
        self.inner.len()
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    pub fn is_empty(&self) -> bool {
        self.inner.is_empty()
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    
    
    
    pub fn clear(&self) {
        self.inner.clear()
    }

    
    pub fn capacity(&self) -> usize {
        self.inner.capacity()
    }

    
    
    /// # Examples
    
    
    
    
    
    
    
    
    pub fn contains<Q>(&self, key: &Q) -> bool
    where
        K: Borrow<Q>,
        Q: Hash + Eq + ?Sized,
    {
        self.inner.contains_key(key)
    }
}

impl<K: Eq + Hash, S: BuildHasher + Clone> IntoIterator for DashSet<K, S> {
    type Item = K;

    type IntoIter = OwningIter<K, S>;

    fn into_iter(self) -> Self::IntoIter {
        OwningIter::new(self.inner.into_iter())
    }
}

impl<K: Eq + Hash, S: BuildHasher + Clone> Extend<K> for DashSet<K, S> {
    fn extend<T: IntoIterator<Item = K>>(&mut self, iter: T) {
        let iter = iter.into_iter().map(|k| (k, ()));

        self.inner.extend(iter)
    }
}

impl<K: Eq + Hash, S: BuildHasher + Clone + Default> FromIterator<K> for DashSet<K, S> {
    fn from_iter<I: IntoIterator<Item = K>>(iter: I) -> Self {
        let mut set = DashSet::default();

        set.extend(iter);

        set
    }
}

#[cfg(test)]
mod tests {
    use crate::DashSet;

    #[test]
    fn test_basic() {
        let set = DashSet::new();

        set.insert(0);

        assert_eq!(set.get(&0).as_deref(), Some(&0));
    }

    #[test]
    fn test_default() {
        let set: DashSet<u32> = DashSet::default();

        set.insert(0);

        assert_eq!(set.get(&0).as_deref(), Some(&0));
    }

    #[test]
    fn test_multiple_hashes() {
        let set = DashSet::<u32>::default();

        for i in 0..100 {
            assert!(set.insert(i));
        }

        for i in 0..100 {
            assert!(!set.insert(i));
        }

        for i in 0..100 {
            assert_eq!(Some(i), set.remove(&i));
        }

        for i in 0..100 {
            assert_eq!(None, set.remove(&i));
        }
    }
}
