#' Print Method for Logic Forest Predictions
#'
#' Displays predictions from a logic forest model, including the predicted classes 
#' and, for classification models, the proportion of trees predicting a class of one.
#'
#' @param x An object of class \code{"LFprediction"}.
#' @param ... Additional arguments (currently ignored).
#'
#' @details
#' For classification models, this method prints the predicted classes for each observation 
#' and the proportion of trees in the logic forest that predict class 1.  
#' For linear regression models, it prints the predicted values and, if available, the out-of-bag mean squared error.
#'
#' @return No return value. This function is called for its side effects (printing).
#'
#' @author Bethany Wolf \email{wolfb@@musc.edu}
#' @seealso \code{\link{predict.logforest}}
#' @export
print.LFprediction<-function(x, ...)
{
  if(inherits(x)!="LFprediction")
    stop("x not of class LFprediction")
  prdct<-x$LFprediction
  if(model.type=="Classification")
  {
    prop<-x$proportion_one

    if(length(x)==3)
    {
      cat("OOB Predicted values\n")
      cat("\n")
      print.default(prdct, quote=FALSE)
      cat("\n")
      cat("Proportion of OOB trees that predict 1")
      cat("\n")
      print.default(prop, quote=FALSE)
    }
    if(length(x)==4)
    {
      cat("Predicted values\n")
      cat("\n")
      print.default(prdct, quote=FALSE)
      cat("\n")
      cat("Proportion of trees that predict 1")
      cat("\n")
      print.default(prop, quote=FALSE)
    }
  }

  if(model.type=="Linear Regression")
  {
    if(length(x)==4)
    {
      cat("OOB Predicted values\n")
      cat("\n")
      print.default(prdct, quote=FALSE)
      cat("\n")
      cat("OOB mean squared error")
      cat("\n")
      print.default(x$OOBmse, quote=FALSE)
    }
    if(length(x)==3)
    {
      cat("Predicted values\n")
      cat("\n")
      print.default(prdct, quote=FALSE)
    }
  }
}
