% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LSMS.R
\name{segm_lsms}
\alias{segm_lsms}
\title{Large-scale segmentation using Mean-Shift}
\usage{
segm_lsms(
  image,
  otb,
  spatialr = 5L,
  ranger = 15,
  minsize = 100L,
  tilesize = 500L,
  mode = "vector",
  ram = 256L
)
}
\arguments{
\item{image}{path to raster, or SpatRaster}

\item{otb}{output of \code{\link[link2GI:linkOTB]{link2GI::linkOTB()}}}

\item{spatialr}{integer. Spatial radius of the neighborhood}

\item{ranger}{range radius defining the radius (expressed in radiometry unit)
in the multispectral space}

\item{minsize}{integer. Minimum size of a region (in pixel unit) in segmentation. Smaller
clusters will be merged to the neighboring cluster with the closest radiometry.
If set to 0 no pruning is done}

\item{tilesize}{integer. Size of the tiles during the tile-wise processing}

\item{mode}{processing mode, either 'vector' or 'raster'. See details}

\item{ram}{integer. Available memory for processing (in MB)}
}
\value{
sf or SpatRaster
}
\description{
Applies the Mean-Shift segmentation algorithm to an image file
or a SpatRaster. Suitable for large images
}
\details{
Mean-Shift is a region-based segmentation algorithm that groups pixels with
similar characteristics. It's a non-parametric clustering technique that groups
pixels based on spatial proximity and feature similarity (color, intensity). This
method is particularly effective for preserving edges and defailt while simplifying
textures in high-resolution images. Steps:
\enumerate{
\item Initialization: Each pixel is treated as a point in a multi-dimensional space
(combining spatial and color features).
\item Mean Shift Iterations: For each pixel, a search window moves toward the region
with the highest data density (local maxima) by calculating the mean of neighboring
pixels within the window.
\item Convergence: The process repeats until the movement of the window becomes
negligible, indicating convergence.
\item Label Assignment: Pixels that converge to the same mode (local maxima) are
grouped into the same region.
}

The most important parameters are:
\itemize{
\item spatialr: defines the size of the neighborhood
\item ranger: determines similarity in the feature space
\item maxiter: limits the number of iterations for convergence
\item thresh: defines the convergence criterion based on pixel movement
}

The processing mode 'vector' will output a vector file, and process the input
image piecewise. This allows performing segmentation of very large images. IN
contrast, 'raster' mode will output a labeled raster, and it cannot handle
large data. If mode is 'raster', all the 'vector_*' arguments are ignored.
}
\examples{
\dontrun{
## load packages
library(link2GI)
library(OTBsegm)
library(terra)

## load sample image
image_sr <- rast(system.file("raster/pnoa.tiff", package = "OTBsegm"))

## connect to OTB (change to your directory)
otblink <- link2GI::linkOTB(searchLocation = "C:/OTB/")

## apply segmentation
results_ms_sf <- segm_lsms(
    image = image_sr,
    otb   = otblink,
    spatialr = 5,
    ranger   = 25,
    minsize  = 10
)

plotRGB(image_sr)
plot(st_geometry(results_ms_sf), add = TRUE)
}
}
