% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EGG.R
\name{EGG}
\alias{EGG}
\title{The four parameter Exponentiated Generalized Gamma family}
\usage{
EGG(mu.link = "log", sigma.link = "log", nu.link = "log", tau.link = "log")
}
\arguments{
\item{mu.link}{defines the mu.link, with "log" link as the default for the mu parameter.}

\item{sigma.link}{defines the sigma.link, with "log" link as the default for the sigma.}

\item{nu.link}{defines the nu.link, with "log" link as the default for the nu parameter.}

\item{tau.link}{defines the tau.link, with "log" link as the default for the tau parameter.}
}
\value{
Returns a gamlss.family object which can be used to fit a EGG distribution in the \code{gamlss()} function.
}
\description{
The four parameter Exponentiated Generalized Gamma distribution
}
\details{
Four parameter Exponentiated Generalized Gamma distribution with parameters \code{mu}, 
\code{sigma}, \code{nu} and \code{tau} has density given by

\eqn{f(x) = \frac{\nu \sigma}{\mu \Gamma(\tau)} \left(\frac{x}{\mu}\right)^{\sigma \tau -1} \exp\left\{ - \left( \frac{x}{\mu} \right)^\sigma \right\} \left\{ \gamma_1\left( \tau, \left( \frac{x}{\mu} \right)^\sigma \right) \right\}^{\nu-1} ,}

for x > 0.
}
\examples{
# Example 1
# Generating some random values with
# known mu, sigma, nu and tau
\donttest{
set.seed(123456)
y <- rEGG(n=100, mu=0.1, sigma=0.8, nu=10, tau=1.5)

# Fitting the model
require(gamlss)

mod <- gamlss(y~1, sigma.fo=~1, nu.fo=~1, tau.fo=~1, 
              family=EGG,
              control=gamlss.control(n.cyc=500, trace=FALSE))

# Extracting the fitted values for mu, sigma, nu and tau
# using the inverse link function
exp(coef(mod, what="mu"))
exp(coef(mod, what="sigma"))
exp(coef(mod, what="nu"))
exp(coef(mod, what="tau"))
}

# Example 2
# Generating random values under some model

\donttest{
# A function to simulate a data set with Y ~ EGG
gendat <- function(n) {
  x1 <- runif(n)
  x2 <- runif(n)
  mu    <- exp(-0.8 + -3 * x1)
  sigma <- exp(0.77 - 2 * x2)
  nu    <- 10
  tau   <- 1.5
  y <- rEGG(n=n, mu=mu, sigma=sigma, nu=nu, tau)
  data.frame(y=y, x1=x1, x2=x2)
}

set.seed(12345)
dat <- gendat(n=200)

mod <- gamlss(y~x1, sigma.fo=~x2, nu.fo=~1, tau.fo=~1, 
              family=EGG, data=dat,
              control=gamlss.control(n.cyc=500, trace=FALSE))

coef(mod, what="mu")
coef(mod, what="sigma")
exp(coef(mod, what="nu"))
exp(coef(mod, what="tau"))
}
}
\references{
Almalki, S. J., & Nadarajah, S. (2014). Modifications of the 
Weibull distribution: A review. Reliability Engineering & 
System Safety, 124, 32-55.

Cordeiro, G. M., Ortega, E. M., & Silva, G. O. (2011). 
The exponentiated generalized gamma distribution with 
application to lifetime data. Journal of statistical 
computation and simulation, 81(7), 827-842.
}
\seealso{
\link{dEGG}
}
\author{
Amylkar Urrea Montoya, \email{amylkar.urrea@udea.edu.co}
}
