% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/samQL.R
\name{samQL}
\alias{samQL}
\title{Training function of Sparse Additive Regression with Quadratic Loss}
\usage{
samQL(
  X,
  y,
  p = 3,
  lambda = NULL,
  nlambda = NULL,
  lambda.min.ratio = 0.005,
  thol = 1e-05,
  max.ite = 1e+05,
  regfunc = "L1"
)
}
\arguments{
\item{X}{Numeric training matrix with \code{n} rows (samples) and
\code{d} columns (features).}

\item{y}{Numeric response vector of length \code{n}.}

\item{p}{The number of basis spline functions. The default value is 3.}

\item{lambda}{Optional user-supplied regularization sequence. If provided,
use a decreasing sequence; warm starts are used along the path and are
usually much faster than fitting a single value.}

\item{nlambda}{The number of lambda values. The default value is 30.}

\item{lambda.min.ratio}{Smallest lambda as a fraction of \code{lambda.max}
(the smallest value that keeps all component functions at zero). The
default is \code{5e-3}.}

\item{thol}{Stopping tolerance. The default value is \code{1e-5}.}

\item{max.ite}{Maximum number of iterations. The default value is \code{1e5}.}

\item{regfunc}{A string indicating the regularizer. The default value is "L1". You can also assign "MCP" or "SCAD" to it.}
}
\value{
\item{p}{
  The number of basis spline functions used in training.
}
\item{X.min}{
  Per-feature minimums from training data (used to rescale test data).
}
\item{X.ran}{
  Per-feature ranges from training data (used to rescale test data).
}
\item{lambda}{
  Sequence of regularization parameters used in training.
}
\item{w}{
  Solution path matrix with size \code{d*p} by \code{length(lambda)}; each
  column corresponds to one regularization parameter.
}
\item{intercept}{
  The solution path of the intercept.
}
\item{df}{
  Degrees of freedom along the solution path (number of non-zero component
  functions).
}
\item{knots}{
  The \code{p-1} by \code{d} matrix. Each column contains the knots applied to the corresponding variable.
}
\item{Boundary.knots}{
  The \code{2} by \code{d} matrix. Each column contains the boundary points applied to the corresponding variable.
}
\item{func_norm}{
  Functional norm matrix (\code{d} by \code{length(lambda)}); each column
  corresponds to one regularization parameter.
}
\item{sse}{
  Sums of square errors of the solution path.
}
}
\description{
Fit a sparse additive regression model with quadratic loss.
}
\details{
The solver combines block coordinate descent, fast iterative
soft-thresholding, and Newton updates. Computation is accelerated by warm
starts and active-set screening.
}
\examples{

## generating training data
n = 100
d = 500
X = 0.5*matrix(runif(n*d),n,d) + matrix(rep(0.5*runif(n),d),n,d)

## generating response
y = -2*sin(X[,1]) + X[,2]^2-1/3 + X[,3]-1/2 + exp(-X[,4])+exp(-1)-1

## Training
out.trn = samQL(X,y)
out.trn

## plotting solution path
plot(out.trn)

## generating testing data
nt = 1000
Xt = 0.5*matrix(runif(nt*d),nt,d) + matrix(rep(0.5*runif(nt),d),nt,d)

yt = -2*sin(Xt[,1]) + Xt[,2]^2-1/3 + Xt[,3]-1/2 + exp(-Xt[,4])+exp(-1)-1

## predicting response
out.tst = predict(out.trn,Xt)
}
\seealso{
\code{\link{SAM}},\code{\link{plot.samQL},\link{print.samQL},\link{predict.samQL}}
}
