# Test SVG detection methods
# These tests verify the correctness of CalSVG_* functions

test_that("CalSVG_MERINGUE returns correct output structure", {
    skip_if_not_installed("RANN")

    set.seed(42)
    n_spots <- 50
    n_genes <- 10

    expr <- matrix(rpois(n_genes * n_spots, lambda = 10),
                   nrow = n_genes, ncol = n_spots)
    rownames(expr) <- paste0("gene_", seq_len(n_genes))
    colnames(expr) <- paste0("spot_", seq_len(n_spots))

    coords <- cbind(x = runif(n_spots), y = runif(n_spots))
    rownames(coords) <- colnames(expr)

    result <- CalSVG_MERINGUE(expr, coords, network_method = "knn",
                              k = 5, verbose = FALSE)

    # Check output structure
    expect_s3_class(result, "data.frame")
    expect_true("gene" %in% names(result))
    expect_true("p.value" %in% names(result))
    expect_true("p.adj" %in% names(result))
    expect_true("observed" %in% names(result))
    expect_equal(nrow(result), n_genes)
})

test_that("CalSVG_binSpect returns correct output structure", {
    skip_if_not_installed("RANN")

    set.seed(42)
    n_spots <- 50
    n_genes <- 10

    expr <- matrix(rpois(n_genes * n_spots, lambda = 10),
                   nrow = n_genes, ncol = n_spots)
    rownames(expr) <- paste0("gene_", seq_len(n_genes))
    colnames(expr) <- paste0("spot_", seq_len(n_spots))

    coords <- cbind(x = runif(n_spots), y = runif(n_spots))
    rownames(coords) <- colnames(expr)

    result <- CalSVG_binSpect(expr, coords, network_method = "knn",
                              k = 5, verbose = FALSE)

    expect_s3_class(result, "data.frame")
    expect_true("gene" %in% names(result))
    expect_true("p.value" %in% names(result))
    expect_true("estimate" %in% names(result))
    expect_equal(nrow(result), n_genes)
})

test_that("CalSVG_SPARKX returns correct output structure", {
    set.seed(42)
    n_spots <- 50
    n_genes <- 10

    expr <- matrix(rpois(n_genes * n_spots, lambda = 10),
                   nrow = n_genes, ncol = n_spots)
    rownames(expr) <- paste0("gene_", seq_len(n_genes))
    colnames(expr) <- paste0("spot_", seq_len(n_spots))

    coords <- cbind(x = runif(n_spots), y = runif(n_spots))
    rownames(coords) <- colnames(expr)

    result <- CalSVG_SPARKX(expr, coords, kernel_option = "single",
                            verbose = FALSE)

    expect_s3_class(result, "data.frame")
    expect_true("gene" %in% names(result))
    expect_true("p.value" %in% names(result))
    expect_true("p.adj" %in% names(result))
    expect_equal(nrow(result), n_genes)
})

test_that("CalSVG unified interface dispatches correctly", {
    skip_if_not_installed("RANN")

    set.seed(42)
    n_spots <- 30
    n_genes <- 5

    expr <- matrix(rpois(n_genes * n_spots, lambda = 10),
                   nrow = n_genes, ncol = n_spots)
    rownames(expr) <- paste0("gene_", seq_len(n_genes))
    colnames(expr) <- paste0("spot_", seq_len(n_spots))

    coords <- cbind(x = runif(n_spots), y = runif(n_spots))
    rownames(coords) <- colnames(expr)

    # Test different methods through unified interface
    result_meringue <- CalSVG(expr, coords, method = "meringue",
                              network_method = "knn", k = 5, verbose = FALSE)
    result_binspect <- CalSVG(expr, coords, method = "binspect",
                              network_method = "knn", k = 5, verbose = FALSE)

    expect_s3_class(result_meringue, "data.frame")
    expect_s3_class(result_binspect, "data.frame")
})

test_that("SVG methods detect spatial patterns", {
    skip_if_not_installed("RANN")

    set.seed(42)
    n_spots <- 150
    n_genes <- 20

    coords <- cbind(x = runif(n_spots, 0, 100), y = runif(n_spots, 0, 100))
    rownames(coords) <- paste0("spot_", seq_len(n_spots))

    # Create expression matrix
    expr <- matrix(0, nrow = n_genes, ncol = n_spots)
    rownames(expr) <- paste0("gene_", seq_len(n_genes))
    colnames(expr) <- rownames(coords)

    # First 5 genes: spatial pattern (strong gradient)
    for (i in 1:5) {
        expr[i, ] <- coords[, "x"] / 5 + rnorm(n_spots, sd = 2) + 10
    }

    # Remaining genes: random
    for (i in 6:n_genes) {
        expr[i, ] <- rnorm(n_spots, mean = 15, sd = 5)
    }

    result <- CalSVG_MERINGUE(expr, coords, network_method = "knn",
                              k = 10, verbose = FALSE)

    # Spatial genes should have lower p-values on average
    spatial_genes <- paste0("gene_", 1:5)
    random_genes <- paste0("gene_", 6:n_genes)

    avg_pval_spatial <- mean(result$p.value[result$gene %in% spatial_genes], 
                             na.rm = TRUE)
    avg_pval_random <- mean(result$p.value[result$gene %in% random_genes], 
                            na.rm = TRUE)

    # Spatial genes should have significantly lower p-values on average
    expect_lt(avg_pval_spatial, avg_pval_random)
})

test_that("p-values are in valid range", {
    skip_if_not_installed("RANN")

    set.seed(42)
    n_spots <- 50
    n_genes <- 10

    expr <- matrix(rpois(n_genes * n_spots, lambda = 10),
                   nrow = n_genes, ncol = n_spots)
    rownames(expr) <- paste0("gene_", seq_len(n_genes))
    colnames(expr) <- paste0("spot_", seq_len(n_spots))

    coords <- cbind(x = runif(n_spots), y = runif(n_spots))
    rownames(coords) <- colnames(expr)

    result <- CalSVG_MERINGUE(expr, coords, verbose = FALSE)

    # All p-values should be between 0 and 1
    expect_true(all(result$p.value >= 0, na.rm = TRUE))
    expect_true(all(result$p.value <= 1, na.rm = TRUE))
    expect_true(all(result$p.adj >= 0, na.rm = TRUE))
    expect_true(all(result$p.adj <= 1, na.rm = TRUE))
})
