% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Spower.R, R/SpowerBatch.R, R/SpowerCurve.R,
%   R/util.R
\name{Spower}
\alias{Spower}
\alias{print.Spower}
\alias{as.data.frame.Spower}
\alias{SpowerBatch}
\alias{print.SpowerBatch}
\alias{as.data.frame.SpowerBatch}
\alias{SpowerCurve}
\alias{interval}
\title{Simulation-based Power Analyses}
\usage{
Spower(
  ...,
  power = NA,
  sig.level = 0.05,
  interval,
  beta_alpha,
  sig.direction = "below",
  replications = 10000,
  integer,
  parallel = FALSE,
  cl = NULL,
  packages = NULL,
  ncores = parallelly::availableCores(omit = 1L),
  predCI = 0.95,
  predCI.tol = 0.01,
  verbose = interactive(),
  check.interval = FALSE,
  maxiter = 150,
  wait.time = NULL,
  lastSpower = NULL,
  select = NULL,
  control = list()
)

\method{print}{Spower}(x, ...)

\method{as.data.frame}{Spower}(x, ...)

SpowerBatch(
  ...,
  interval = NULL,
  power = NA,
  sig.level = 0.05,
  beta_alpha = NULL,
  sig.direction = "below",
  replications = 10000,
  integer,
  fully.crossed = TRUE,
  parallel = FALSE,
  cl = NULL,
  ncores = parallelly::availableCores(omit = 1L),
  predCI = 0.95,
  predCI.tol = 0.01,
  verbose = interactive(),
  check.interval = FALSE,
  maxiter = 150,
  wait.time = NULL,
  select = NULL,
  control = list()
)

\method{print}{SpowerBatch}(x, ...)

\method{as.data.frame}{SpowerBatch}(x, ...)

SpowerCurve(
  ...,
  interval = NULL,
  power = NA,
  sig.level = 0.05,
  sig.direction = "below",
  replications = 2500,
  integer,
  plotCI = TRUE,
  plotly = TRUE,
  parallel = FALSE,
  cl = NULL,
  ncores = parallelly::availableCores(omit = 1L),
  predCI = 0.95,
  predCI.tol = 0.01,
  verbose = interactive(),
  check.interval = FALSE,
  maxiter = 50,
  wait.time = NULL,
  select = NULL,
  batch = NULL,
  control = list()
)

interval(lower, upper, integer, check.interval = FALSE)
}
\arguments{
\item{...}{expression to use in the simulation that returns a \code{numeric}
vector containing either the p-value (under the null hypothesis), the
probability of the alternative hypothesis in the Bayesian setting,
where the first numeric value
in this vector is treated as the focus for all analyses other than
prospective/post-hoc power.
This corresponds to the \code{alpha} value used to flag samples
as 'significant' when evaluating the null hypothesis
(via p-values; \eqn{P(D|H_0)}),
where any returned p-value less that \code{sig.level} indicates
significance. However, if \code{sig.direction = 'above'} then only values
above \code{sig.level} are flagged as significant, which is useful
in Bayesian posterior probability contexts that focus on the alternative
hypothesis, \eqn{P(H_1|D)}.

Alternatively, a \code{logical} vector can be returned (e.g., when using
confidence intervals (CIs) or evaluating regions of practical
equivalence (ROPEs)), where the average of these
TRUE/FALSE vector corresponds to the empirical power.

Finally, a named \code{list} or \code{data.frame} can be returned instead
if there is need for more general, heterogeneous objects, however
a specific element to extract must be specified
using the \code{select} argument to indicate which of the list elements
are to be used in the power computations. All other elements from the
simulation can, however, be
extracted from the \code{\link[SimDesign]{SimResults}} function.

For \code{\link{SpowerCurve}} and \code{\link{SpowerBatch}},
first expression input must be identical to \code{...} in
\code{\link{Spower}}, while the remaining named inputs must match
the arguments to this expression to indicate which variables should
be modified in the resulting power curves. Providing \code{NA} values
is also supported to solve the missing component.  Note that only the
first three named arguments in \code{\link{SpowerCurve}} will be plotted
using the x-y, colour, and facet wrap aesthetics, respectively. However,
if necessary the data can be extracted for further visualizations via
\code{\link[ggplot2]{ggplot_build}} to provide more customized control}

\item{power}{power level to use. If set to \code{NA} (default)
then the empirical power  will be estimated given the fixed \code{...}
inputs (e.g., for prospective/post-hoc power analysis). For
\code{\link{SpowerCurve}} and \code{\link{SpowerBatch}} this can be
a vector}

\item{sig.level}{alpha level to use (default is \code{.05}).
If set to \code{NA} then the value will
be estimated given the fixed \code{conditions} input
(e.g., for criterion power analysis). Only used when the value returned
from the experiment is a \code{numeric} (e.g., a p-value, or a
posterior probability; see \code{sig.direction}).

If the return of the supplied experiment is a
\code{logical} then this argument will be entirely ignored. As such,
arguments such as \code{conf.level} should be included
in the simulation experiment definition itself
to indicate the explicit inferential criteria, and so that this
argument can be manipulated should the need arise.}

\item{interval}{required search interval to use when
\code{\link[SimDesign]{SimSolve}} is called to perform stochastic
root solving. Note that for compromise analyses, where the
\code{sig.level} is set to \code{NA}, if not set explicitly then the
interval will default to \code{c(0,1)}.

Alternatively, though only for the function \code{Spower()},
the function \code{\link{interval}} can be used within the
experiment function definition where the canonical \code{NA} placeholder
is used. Arguments from \code{\link{interval}} will then be extracted and
passed to \code{\link{Spower}} as usual. Note that this is not supported
in \code{SpowerBatch} and \code{SpowerCurve} as multiple interval
definitions are often required; hence, \code{NA} placeholders
are always required in these wrapper functions}

\item{beta_alpha}{(optional) ratio to use in compromise analyses
corresponding to the Type II errors (beta) over the Type I error (alpha).
Ratios greater than \eqn{q = \beta/\alpha = 1} indicate that Type I
errors are worse than Type II, while ratios
less than one the opposite. A ratio equal to 1 gives an equal trade-off
between Type I and Type II errors}

\item{sig.direction}{a character vector that is either \code{'below'}
(default) or \code{'above'} to indicate which direction relative to
\code{sig.level} is considered significant. This is useful, for instance,
when forming cutoffs for Bayesian
posterior probabilities organized to show support
for the hypothesis of interest (\eqn{P(H_1|D)}). As an example,
setting \code{sig.level = .95} with \code{sig.direction = 'above'}
flags a sample as 'significant' whenever the
posterior probability is greater than .95.}

\item{replications}{number of replications to use when
\code{\link[SimDesign]{runSimulation}} is required. Default is 10000,
though set to 2500 for \code{\link{SpowerCurve}}}

\item{integer}{a logical value indicating whether the search iterations
use integers or doubles.

If missing, automatically set to \code{FALSE} if \code{interval} contains
non-integer numbers or the range is less than 5, as well as
when \code{sig.level = NA}}

\item{parallel}{for parallel computing for slower simulation experiments
(see \code{\link[SimDesign]{runSimulation}} for details).}

\item{cl}{see \code{\link[SimDesign]{runSimulation}}}

\item{packages}{see \code{\link[SimDesign]{runSimulation}}}

\item{ncores}{see \code{\link[SimDesign]{runSimulation}}}

\item{predCI}{predicting confidence interval level
(see \code{\link[SimDesign]{SimSolve}})}

\item{predCI.tol}{predicting confidence interval consistency tolerance
for stochastic root solver convergence
(see \code{\link[SimDesign]{SimSolve}}).
Default converges when the power rate CI is consistently
within \code{.01/2} of the target power}

\item{verbose}{logical; should information be printed to the console?
By default this is determined based on whether the session is interactive
or not}

\item{check.interval}{logical; check the interval range validity
(see \code{\link[SimDesign]{SimSolve}}). Disabled by default}

\item{maxiter}{maximum number of stochastic root-solving iterations.
Default is 150, though set to 50 for \code{\link{SpowerCurve}}}

\item{wait.time}{(optional) argument to indicate the time to wait
(specified in minutes if supplied as a numeric vector).
See \code{\link[SimDesign]{SimSolve}} for details and
See \code{\link[SimDesign]{timeFormater}} for further specifications}

\item{lastSpower}{a previously returned \code{Spower} object to be updated.
Use this if you want to continue where an estimate left off but wish to
increase the precision (e.g., by adding more replications, or by letting
the stochastic root solver continue searching).

Note that if the object was not stored use \code{\link{getLastSpower}}
to obtain the last estimated power object}

\item{select}{a character vector indicating which elements to
extract from the provided stimulation experiment function. By default,
all elements from the provided function will be used, however if the
provided function contains information not relevant to the power
computations (e.g., parameter estimates, standard errors, etc) then these
should be ignored. To extract the complete results post-analysis use
\code{\link[SimDesign]{SimResults}} to allow manual summarizing of the
stored results (applicable only with prospective/post-hoc power)}

\item{control}{a list of control parameters to pass to
\code{\link[SimDesign]{runSimulation}} or \code{\link[SimDesign]{SimSolve}}}

\item{x}{object of class \code{'Spower'}. If \code{\link{SpowerBatch}} were used
the this will be a \code{list}}

\item{fully.crossed}{logical; should the supplied conditions to
\code{SpowerBatch} be fully crossed? Passed to the same
argument documented in \code{\link[SimDesign]{createDesign}}}

\item{plotCI}{logical; include confidence/predicted confidence intervals
in plots?}

\item{plotly}{logical; draw the graphic into the interactive \code{plotly}
interface? If \code{FALSE} the ggplot2 object will be returned instead}

\item{batch}{if \code{\link{SpowerBatch}} were previously used to perform
the computations
then this information can be provided to this \code{batch} argument to
avoid recomputing}

\item{lower}{lower bound for stochastic search interval. If input
contains a decimal then \code{Spower(..., integer)} will be
set to \code{FALSE}}

\item{upper}{upper bound for stochastic search interval. If input
contains a decimal then \code{Spower(..., integer)} will be
set to \code{FALSE}}
}
\value{
an invisible \code{tibble}/\code{data.frame}-type object of
class \code{'Spower'} containing the power results from the
simulation experiment

a ggplot2 object automatically rendered with
\code{plotly} for interactivity
}
\description{
General purpose function that provides power-focused estimates for
a priori, prospective/post-hoc, compromise, sensitivity, and criterion
power analysis. Function provides a general wrapper to the
\code{SimDesign} package's \code{\link[SimDesign]{runSimulation}} and
\code{\link[SimDesign]{SimSolve}} functions. As such, parallel processing is
automatically supported, along with progress bars,
confidence/predicted confidence intervals for the results estimates,
safety checks, and more.

The function \code{\link{SpowerBatch}}, on the other hand, can be used to
run \code{\link{Spower}} across
different simulation combinations, returning a \code{list} of results instead.
Can also be used as a pre-computing step before using
\code{\link{SpowerCurve}}, and shares the same syntax specification (see
\code{\link{SpowerCurve}} for further examples).

\code{\link{SpowerCurve}} draws power curves that either a) estimate the
power given a
set of varying conditions or b) solves a set of root conditions
given fixed values of power. Confidence/predicted confidence intervals are
included in the output to reflect the estimate uncertainties, though note
that fewer replications/iterations are used compared to
\code{\link{Spower}} as the goal is visualization of competing
variable inputs rather than precision of a given input.
}
\details{
Five types of power analysis flavors can be performed with \code{Spower},
which are triggered based on which supplied input is set to
missing (\code{NA}):

\describe{
\item{A Priori}{Solve for a missing sample size component
(e.g., \code{n}) to achieve a specific target power rate}
\item{Prospective and Post-hoc}{Estimate the power rate given a set of
fixed conditions. If estimates of effect sizes and other empirical
characteristics (e.g., observed sample size) are supplied this results
in observed/retrospective power (not recommended), while if only
sample size is included as the observed quantity, but the effect
sizes are treated as unknown, then this results in post-hoc
power (Cohen, 1988)}
\item{Sensitivity}{Solve a missing effect size value as a function of
the other supplied constant components}
\item{Criterion}{Solve the error rate (argument \code{sig.level}) as a
function of the other supplied constant components}
\item{Compromise}{Solve a Type I/Type II error trade-off ratio as a
function of the other supplied constant components and the
target ratio \eqn{q = \beta/\alpha} (argument \code{beta_alpha})}
}

To understand how the package is structured, the first expression in
the \code{...} argument, which contains the simulation experiment
definition for a single sample,
is passed to either \code{\link[SimDesign]{SimSolve}} or
\code{\link[SimDesign]{runSimulation}} depending on which element (including
the \code{power} and \code{sig.level} arguments) is set to \code{NA}.
For instance, \code{Spower(p_t.test(n=50, d=.5))} will perform a
prospective/post-hoc power evaluation since \code{power = NA} by default,
while \code{Spower(p_t.test(n=NA, d=.5), power = .80)} or,
equivalently, \code{Spower(p_t.test(n=interval(.,.), d=.5), power = .80)},
will perform an a priori power analysis to solve the missing
\code{n} argument.

For expected power computations, the arguments to the simulation
experiment arguments can be specified as a function to reflect
the prior uncertainty. For instance, if
\code{d_prior <- function() rnorm(1, mean=.5, sd=1/8)} then
\code{Spower(p_t.test(n=50, d=d_prior())} will compute the expected power
over the prior sampling distribution for \code{d}
}
\examples{

############################
# Independent samples t-test
############################

# Internally defined p_t.test function
args(p_t.test)    # missing arguments required
# help(p_t.test)  # additional information

# p_* functions generate data and return single p-value
p_t.test(n=50, d=.5)
p_t.test(n=50, d=.5)

# test that it works
Spower(p_t.test(n = 50, d = .5), replications=10)

# also behaves naturally with a pipe
p_t.test(n = 50, d = .5) |> Spower(replications=10)

\donttest{

# Estimate power given fixed inputs (prospective power analysis)
out <- Spower(p_t.test(n = 50, d = .5))
summary(out)   # extra information
as.data.frame(out)  # coerced to data.frame

# increase precision (not run)
# p_t.test(n = 50, d = .5) |> Spower(replications=30000)

# alternatively, increase precision from previous object.
#   Here we add 20000 more replications on top of the previous 10000
p_t.test(n = 50, d = .5) |>
  Spower(replications=20000, lastSpower=out) -> out2
out2$REPLICATIONS  # total of 30000 replications for estimate

# previous analysis not stored to object, but can be retrieved
out <- getLastSpower()
out   # as though it were stored from Spower()

# Same as above, but executed with multiple cores (not run)
p_t.test(n = 50, d = .5) |>
   Spower(replications=30000, parallel=TRUE, ncores=2)

# Solve N to get .80 power (a priori power analysis)
p_t.test(n = interval(2,500), d = .5) |> Spower(power=.8) -> out
summary(out)  # extra information
plot(out)
plot(out, type = 'history')

# total sample size required
ceiling(out$n) * 2

# equivalently, using NA within the experiment definition
p_t.test(n = NA, d = .5) |> Spower(power=.8, interval=c(2,500))

# same as above, but in parallel with 2 cores
out.par <- p_t.test(n = interval(2,500), d = .5) |>
  Spower(power=.8, parallel=TRUE, ncores=2)
summary(out.par)

# similar information from pwr package
(pwr <- pwr::pwr.t.test(d=.5, power=.80))
ceiling(pwr$n) * 2

# If greater precision is required and the user has a specific amount of
# time they are willing to wait (e.g., 5 minutes) then wait.time can be used.
# Below estimates root after searching for 1 minute, and run in parallel
# with 2 cores (not run)
p_t.test(n = interval(2,500), d = .5) |>
  Spower(power=.8, wait.time='1', parallel=TRUE, ncores=2)

# Similar to above for precision improvements, however letting
#  the root solver continue searching from an early search history.
#  Usually a good idea to increase the maxiter and lower the predCI.tol
p_t.test(n = interval(2,500), d = .5) |>
  Spower(power=.8, lastSpower=out,
        maxiter=200, predCI.tol=.008) #starts at last iteration in "out"

# Solve d to get .80 power (sensitivity power analysis)
p_t.test(n = 50, d = interval(.1, 2)) |> Spower(power=.8)
pwr::pwr.t.test(n=50, power=.80) # compare

# Solve alpha that would give power of .80 (criterion power analysis)
#    interval not required (set to interval = c(0, 1))
p_t.test(n = 50, d = .5) |> Spower(power=.80, sig.level=NA)

# Solve beta/alpha ratio to specific error trade-off constant
#   (compromise power analysis)
out <- p_t.test(n = 50, d = .5) |> Spower(beta_alpha = 2)
with(out, (1-power)/sig.level)   # solved ratio

# update beta_alpha criteria without re-simulating
(out2 <- update(out, beta_alpha=4))
with(out2, (1-power)/sig.level)   # solved ratio

##############
# Power Curves
##############

# SpowerCurve() has similar input, though requires varying argument
p_t.test(d=.5) |> SpowerCurve(n=c(30, 60, 90))

# solve n given power and plot
p_t.test(n=NA, d=.5) |> SpowerCurve(power=c(.2, .5, .8), interval=c(2,500))

# multiple varying components
p_t.test() |> SpowerCurve(n=c(30,60,90), d=c(.2, .5, .8))

################
# Expected Power
################

# Expected power computed by including effect size uncertainty.
# For instance, belief is that the true d is somewhere around d ~ N(.5, 1/8)
dprior <- function(x, mean=.5, sd=1/8) dnorm(x, mean=mean, sd=sd)
curve(dprior, -1, 2, main=expression(d \%~\% N(0.5, 1/8)),
      xlab='d', ylab='density')

# For Spower, define prior sampler for specific parameter(s)
d_prior <- function() rnorm(1, mean=.5, sd=1/8)
d_prior(); d_prior(); d_prior()

# Replace d constant with d_prior to compute expected power
p_t.test(n = 50, d = d_prior()) |> Spower()

# A priori power analysis using expected power
p_t.test(n = interval(2,500), d = d_prior()) |> Spower(power=.8)
pwr::pwr.t.test(d=.5, power=.80) # expected power result higher than fixed d


###############
# Customization
###############

#   Make edits to the function for customization
if(interactive()){
    p_my_t.test <- edit(p_t.test)
    args(p_my_t.test)
    body(p_my_t.test)
}

# Alternatively, define a custom function (potentially based on the template)
p_my_t.test <- function(n, d, var.equal=FALSE, n2_n1=1, df=10){

    # Welch power analysis with asymmetric distributions
    # group2 as large as group1 by default

    # degree of skewness controlled via chi-squared distribution's df
    group1 <- rchisq(n, df=df)
    group1 <-  (group1 - df) / sqrt(2*df)   # Adjusted mean to 0, sd = 1
    group2 <- rnorm(n*n2_n1, mean=d)
    dat <- data.frame(group = factor(rep(c('G1', 'G2'),
                                     times = c(n, n*n2_n1))),
    				  DV = c(group1, group2))
    obj <- t.test(DV ~ group, dat, var.equal=var.equal)
    p <- obj$p.value
    p
}

# Solve N to get .80 power (a priori power analysis), using defaults
p_my_t.test(n = interval(2,500), d = .5, n2_n1=2) |>
  Spower(power=.8) -> out

# total sample size required
with(out, ceiling(n) + ceiling(n * 2))

# Solve N to get .80 power (a priori power analysis), assuming
#   equal variances, group2 2x as large as group1, large skewness
p_my_t.test(n = interval(30,100), d=.5, var.equal=TRUE, n2_n1=2, df=3) |>
  Spower(power=.8) -> out2

# total sample size required
with(out2, ceiling(n) + ceiling(n * 2))

# prospective power, can be used to extract the adjacent information
p_my_t.test(n = 100, d = .5) |> Spower() -> post

###############################
# Using CIs instead of p-values
###############################

# CI test returning TRUE if psi0 is outside the 95\% CI
ci_ind.t.test <- function(n, d, psi0=0, conf.level=.95){
  g1 <- rnorm(n)
  g2 <- rnorm(n, mean=d)
  CI <- t.test(g2, g1, var.equal=TRUE,conf.level=conf.level)$conf.int
  is.outside_CI(psi0, CI)
}

# returns logical
ci_ind.t.test(n=100, d=.2)
ci_ind.t.test(n=100, d=.2)

# simulated prospective power
ci_ind.t.test(n=100, d=.2) |> Spower()

# compare to pwr package
pwr::pwr.t.test(n=100, d=.2)

############################
# Equivalence test power using CIs
#
# H0: population d is outside interval [LB, UB] (not tolerably equivalent)
# H1: population d is within interval [LB, UB]  (tolerably equivalent)

# CI test returning TRUE if CI is within tolerable equivalence range (tol)
ci_equiv.t.test <- function(n, d, tol, conf.level=.95){
  g1 <- rnorm(n)
  g2 <- rnorm(n, mean=d)
  CI <- t.test(g2, g1, var.equal=TRUE,conf.level=conf.level)$conf.int
  is.CI_within(CI, tol)
}

# evaluate if CI is within tolerable interval (tol)
ci_equiv.t.test(n=1000, d=.2, tol=c(.1, .3))

# simulated prospective power
ci_equiv.t.test(n=1000, d=.2, tol=c(.1, .3)) |> Spower()

# higher power with larger N (more precision) or wider tol interval
ci_equiv.t.test(n=2000, d=.2, tol=c(.1, .3)) |> Spower()
ci_equiv.t.test(n=1000, d=.2, tol=c(.1, .5)) |> Spower()

####
# superiority test (one-tailed)
# H0: population d is less than LB    (not superior)
# H1: population d is greater than LB (superior)

# set upper bound to Inf as it's not relevant, and reduce conf.level
#   to reflect one-tailed test
ci_equiv.t.test(n=1000, d=.2, tol=c(.1, Inf), conf.level=.90) |>
  Spower()

# higher LB means greater requirement for defining superiority (less power)
ci_equiv.t.test(n=1000, d=.2, tol=c(.15, Inf), conf.level=.90) |>
  Spower()

}

##############################################
# SpowerBatch() examples
##############################################

\dontrun{

# estimate power given varying sample sizes
p_t.test(d=0.2) |>
  SpowerBatch(n=c(30, 90, 270, 550), replications=1000) -> nbatch
nbatch

# can be stacked to view the output as data.frame
as.data.frame(nbatch)

# plot with SpowerCurve()
SpowerCurve(batch=nbatch)

# equivalent, but re-runs the computations
p_t.test(d=0.2) |> SpowerCurve(n=c(30, 90, 270, 550), replications=1000)

# estimate power given varying sample sizes and effect size
p_t.test() |> SpowerBatch(n=c(30, 90, 270, 550),
                          d=c(.2, .5, .8), replications=1000) -> ndbatch
ndbatch

# plot with SpowerCurve()
SpowerCurve(batch=ndbatch)

# For non-crossed experimental combinations, pass fully.crossed = FALSE. Note
# that this requires the lengths of the inputs to match
p_t.test() |> SpowerBatch(n=c(30, 90, 270),
           d=c(.2, .5, .8), replications=1000, fully.crossed=FALSE) -> batch3


##############################

# Batches also useful for drawing graphics outside of current framework
# in SpowerCurve(). Below an image is drawn pertaining to the distribution
# of the effects (H0 vs Ha hypotheses), giving the classic sampling
# distribution comparisons of the effect sizes, however presents the
# information using kernel density plots as this may be useful when the
# sampling distributions are non-normal

# Define wrapper function that returns p-value and estimated mean difference
Ice_T <- function(...){
	  out <- p_t.test(..., return_analysis=TRUE)
	  ret <- c(p=out$p.value, mu_d=unname(with(out, estimate[1] - estimate[2])))
   ret
}

# rapper returns p-value and effect size of interest
Ice_T(n=90, d=.5)

# run batch mode to get 4 mean difference combinations, selecting out only
# the 'p' for the power-analysis portions
batch <- Ice_T(n=90) |>
   SpowerBatch(d=c(0, .2, .5, .8), select="p")
batch
as.data.frame(batch)

# create big table of results across the batches
results <- SimResults(batch, rbind=TRUE)
results$d <- factor(results$d)
results

# draw H0 vs Ha relationships for each effect size
library(ggplot2)
library(patchwork)
gg1 <- ggplot(subset(results, d \%in\% c(0, .2)),
			  aes(mu_d, colour=d)) +
	geom_density() + ggtitle('Small effect (d = 0.2)') +
	theme(legend.position='none') +
	xlab(expression(mu[d])) + xlim(c(-0.75, 1.5))
gg2 <- ggplot(subset(results, d \%in\% c(0, .5)),
			  aes(mu_d, colour=d)) +
	  geom_density() + ggtitle('Medium effect (d = 0.5)') +
	  theme(legend.position='none') + xlab(expression(mu[d])) +
	  xlim(c(-0.75, 1.5))
gg3 <- ggplot(subset(results, d \%in\% c(0, .8)),
			  aes(mu_d, colour=d)) +
	  geom_density() + ggtitle('Large effect (d = 0.8)') +
	  theme(legend.position='none') + xlab(expression(mu[d])) +
	  xlim(c(-0.75, 1.5))

gg1 / gg2 / gg3

}


\donttest{

##############################################
# SpowerCurve() examples
##############################################

# estimate power given varying sample sizes
gg <- p_t.test(d=0.2) |> SpowerCurve(n=c(30, 90, 270, 550))

# Output ggplot2 object (rendered with plotly); hence, can be modified
library(ggplot2)
gg + geom_text(aes(label=power), size=5, colour='red', nudge_y=.05) +
  ylab(expression(1-beta)) + theme_grey()

# Increase precision by using 10000 replications. Parallel computations
#   generally recommended in this case to save time
p_t.test(d=0.2) |> SpowerCurve(n=c(30, 90, 270, 550), replications=10000)

# estimate sample sizes given varying power
p_t.test(n=NA, d=0.2) |>
  SpowerCurve(power=c(.2, .4, .6, .8), interval=c(10, 1000))

# get information from last printed graphic instead of saving
gg <- last_plot()
gg + coord_flip() # flip coordinates to put power on y-axis

# estimate power varying d
p_t.test(n=50) |> SpowerCurve(d=seq(.1, 1, by=.2))

# estimate d varying power
p_t.test(n=50, d=NA) |>
  SpowerCurve(power=c(.2, .4, .6, .8), interval=c(.01, 1))


#####

# vary two inputs instead of one (second input uses colour aesthetic)
p_t.test() |> SpowerCurve(n=c(30, 90, 270, 550),
                         d=c(.2, .5, .8))

# extract data for alternative presentations
build <- ggplot_build(last_plot())
build

df <- build$plot$data
head(df)
ggplot(df, aes(n, power, linetype=d)) + geom_line()

# vary three arguments (third uses facet_wrap ... any more than that and
#   you're on your own!)
p_t.test() |> SpowerCurve(n=c(30, 90, 270, 550),
                         d=c(.2, .5, .8),
                         var.equal=c(FALSE, TRUE))

########################################

# If objects were precomputed using SpowerBatch() then
#  these can be plotted instead
p_t.test(d=0.2) |>
  SpowerBatch(n=c(30, 90, 270, 550), replications=1000) -> nbatch
nbatch
as.data.frame(nbatch)

# plot the results, but avoid further computations
SpowerCurve(batch=nbatch)

}

}
\seealso{
\code{\link{update}}, \code{\link{SpowerCurve}},
\code{\link{getLastSpower}}, \code{\link{is.CI_within}},
\code{\link{is.outside_CI}}

\code{\link{Spower}}, \code{\link{SpowerBatch}}
}
\author{
Phil Chalmers \email{rphilip.chalmers@gmail.com}
}
