test_that("extract_benchmark_metrics works correctly", {
  set.seed(123)
  data <- data.frame(
    patient_id = 1:200,
    age = rnorm(200, 50, 15),
    cost = rnorm(200, 10000, 5000),
    is_error = sample(c(0, 1), 200, replace = TRUE, prob = c(0.95, 0.05))
  )
  
  error_idx <- which(data$is_error == 1)
  data$cost[error_idx] <- data$cost[error_idx] * 10
  
  scored_data <- score_anomaly(data, ground_truth_col = "is_error")
  metrics <- extract_benchmark_metrics(scored_data)
  
  expect_false(is.null(metrics))
  expect_true("auc_roc" %in% names(metrics))
  expect_true("auc_pr" %in% names(metrics))
  expect_true("top_k_recall" %in% names(metrics))
  expect_true(is.numeric(metrics$auc_roc))
})

test_that("extract_benchmark_metrics warns when no metrics available", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000)
  )
  
  scored_data <- score_anomaly(data)
  
  expect_warning(extract_benchmark_metrics(scored_data))
})

test_that("get_top_anomalies works correctly", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15),
    cost = rnorm(100, 10000, 5000)
  )
  
  scored_data <- score_anomaly(data)
  top_10 <- get_top_anomalies(scored_data, n = 10)
  
  expect_true(is.data.frame(top_10))
  expect_true(nrow(top_10) == 10)
  expect_true(all(diff(top_10$anomaly_score) <= 0))  # Should be sorted descending
})

test_that("get_top_anomalies handles n larger than dataset", {
  data <- data.frame(
    patient_id = 1:50,
    age = rnorm(50, 50, 15),
    cost = rnorm(50, 10000, 5000)
  )
  
  scored_data <- score_anomaly(data)
  top_100 <- get_top_anomalies(scored_data, n = 100)
  
  expect_true(nrow(top_100) == 50)  # Should return all records
})

test_that("get_top_anomalies errors on invalid inputs", {
  data <- data.frame(
    patient_id = 1:100,
    age = rnorm(100, 50, 15)
  )
  
  expect_error(get_top_anomalies(data))  # No anomaly_score
  expect_error(get_top_anomalies("not a data frame"))
})

