test_triangle_1 <- make_test_triangle(data = matrix(
  c(
    65, 46, 21, 7,
    70, 40, 20, 5,
    80, 50, 10, 10,
    100, 40, 31, 20,
    95, 45, 21, NA,
    82, 42, NA, NA,
    70, NA, NA, NA
  ),
  nrow = 7,
  byrow = TRUE
))

test_triangle_2 <- make_test_triangle(data = matrix(
  c(
    65, 46, 21, 7,
    70, 40, 20, 5,
    80, 50, 10, 10,
    100, 40, 31, NA,
    95, 45, NA, NA,
    82, NA, NA, NA
  ),
  nrow = 6,
  byrow = TRUE
))

retro_rts_list <- list(test_triangle_1, test_triangle_2)
delay_pmf <- c(0.1, 0.1, 0.5, 0.3)

test_that("estimate_and_apply_delays returns correctly structured output", {
  result <- estimate_and_apply_delays(
    retro_reporting_triangles = retro_rts_list
  )

  # Output has same number of elements as input
  expect_length(result, 2)

  # Each output matrix has same dimensions as input
  for (i in seq_along(retro_rts_list)) {
    expect_identical(dim(result[[i]]), dim(retro_rts_list[[i]]))
  }

  # Check that NAs are replaced (lower-right corner filled)
  # This assumes your method removes all NAs
  expect_false(anyNA(result[[1]]))
  expect_false(anyNA(result[[2]]))
})

test_that("estimate_and_apply_delays takes in delay_pmf as vector or list", {
  result1 <- estimate_and_apply_delays(
    retro_reporting_triangles = retro_rts_list,
    delay_pmf = delay_pmf
  )

  result2 <- estimate_and_apply_delays(
    retro_reporting_triangles = retro_rts_list,
    delay_pmf = rep(list(delay_pmf), length(retro_rts_list))
  )

  expect_identical(result1, result2)

  expect_error(estimate_and_apply_delays(
    retro_reporting_triangles = retro_rts_list,
    delay_pmf = rep(list(delay_pmf), length(retro_rts_list) - 1)
  ))
})

test_that(
  "estimate_and_apply_delays default n_history_delay uses minimum rows",
  {
    # Input matrices have 7 and 7 rows → min = 6
    result_default <- estimate_and_apply_delays(
      retro_reporting_triangles = retro_rts_list
    )
    result_custom_w_def <- estimate_and_apply_delays(
      retro_reporting_triangles = retro_rts_list,
      n = 6
    )
    result_custom <- estimate_and_apply_delays(
      retro_reporting_triangles = retro_rts_list,
      n = 5
    )

    # Ensure default matches explicit use of min rows
    expect_identical(result_default, result_custom_w_def)
    # Because estimates are made from different observations, expect these
    # not to be identical
    expect_false(all(result_default[[1]] == result_custom[[1]]))
  }
)

test_that("estimate_and_apply_delays custom n_history_delay is respected", {
  result <- estimate_and_apply_delays(
    retro_reporting_triangles = retro_rts_list,
    n = 5
  )
  # Compare to estimate using n=5
  first_triangle <- result[[1]]
  delay_pmf <- estimate_delay(retro_rts_list[[1]],
    n = 5
  )
  exp_first_triangle <- apply_delay(retro_rts_list[[1]], delay_pmf)
  expect_equal(first_triangle, exp_first_triangle, tol = 0.0001)

  # Custom n_history_delay is too high
  expect_error(
    estimate_and_apply_delays(
      retro_reporting_triangles = retro_rts_list,
      n = 8
    )
  ) # nolint
  # Custom n_history_delay is too low
  expect_error(
    estimate_and_apply_delays(
      retro_reporting_triangles = retro_rts_list,
      n = 3
    )
  )
})

test_that("estimate_and_apply_delays invalid inputs throw errors", {
  # Non-list input
  expect_error(estimate_and_apply_delays(
    retro_reporting_triangles = "not_a_list"
  ))

  # Invalid n_history_delay values
  expect_error(estimate_and_apply_delays(retro_rts_list, n = -1))
  expect_error(estimate_and_apply_delays(retro_rts_list, n = "two"))
})

test_that("estimate_and_apply_delays identical-sized matrices work", {
  same_size_list <- list(test_triangle_1[2:7, ], test_triangle_2)
  result <- estimate_and_apply_delays(same_size_list)

  # Number of rows of each matrix should be identical (6 and 6)
  expect_identical(sapply(result, nrow), c(6L, 6L))
})

test_that("estimate_and_apply_delays handles a single triangle with 0s for first column appropriately", { # nolint

  triangle3 <- matrix(
    c(
      0, 40, 20, 5,
      0, 50, 10, 10,
      0, 40, 31, NA,
      0, 45, NA, NA,
      0, NA, NA, NA
    ),
    nrow = 5,
    byrow = TRUE
  )

  retro_rts_list <- list(test_triangle_1, test_triangle_2, triangle3)

  result <- expect_message(estimate_and_apply_delays(retro_rts_list))
  expect_null(result[[3]])
})

test_that("estimate_and_apply_delays errors if only contains triangles with first column 0", { # nolint
  triangle1 <- matrix(
    c(
      0, 46, 21, 7,
      0, 40, 20, 5,
      0, 50, 10, 10,
      0, 40, 31, 20,
      0, 45, 21, NA,
      0, 42, NA, NA,
      0, NA, NA, NA
    ),
    nrow = 7,
    byrow = TRUE
  )

  triangle2 <- matrix(
    c(
      0, 46, 21, 7,
      0, 40, 20, 5,
      0, 50, 10, 10,
      0, 40, 31, NA,
      0, 45, NA, NA,
      0, NA, NA, NA
    ),
    nrow = 6,
    byrow = TRUE
  )
  triangle3 <- matrix(
    c(
      0, 40, 20, 5,
      0, 50, 10, 10,
      0, 40, 31, NA,
      0, 45, NA, NA,
      0, NA, NA, NA
    ),
    nrow = 5,
    byrow = TRUE
  )

  retro_rts_list <- list(triangle1, triangle2, triangle3)

  expect_error(estimate_and_apply_delays(retro_rts_list))
})

test_that("estimate_and_apply_delays uses full number of rows in n_history_delay", { # nolint
  sim_delay_pmf <- c(0.4, 0.3, 0.2, 0.1)

  # Generate counts for each reference date
  counts <- c(100, 150, 200, 250, 300, 100, 90)

  # Create a complete triangle based on the known delay PMF
  complete_triangle <- lapply(counts, function(x) x * sim_delay_pmf)
  complete_triangle <- make_test_triangle(
    data = do.call(rbind, complete_triangle)
  )

  check_pmf <- estimate_delay(complete_triangle, n = 7)
  check_pmf

  # Create a reporting triangle with NAs in the lower right
  triangle <- apply_reporting_structure(complete_triangle)
  triangle

  slight_dif_triangle <- estimate_and_apply_delay(triangle, n = 7)

  expect_equal(unclass(slight_dif_triangle), unclass(complete_triangle),
    tol = 0.5, check.attributes = FALSE
  )
  truncated_rts <- truncate_to_rows(triangle, n = 2)
  truncated_rts[1:2]
  # These will always have the first row at the top. First one will be with
  # last row cut off, second will be with last 2 rows cut off

  retro_rts <- apply_reporting_structures(truncated_rts)


  # Now change n_history_delay to only use last 4 rows, we should never use
  # first row so both will be equal
  retro_pt_nowcast_mat_list <- estimate_and_apply_delays(
    retro_reporting_triangles = retro_rts,
    n = 4
  )

  pmf_list <- lapply(retro_pt_nowcast_mat_list, estimate_delay)
  # Both are returning the original pmf bc they dont use the modified one
  expect_equal(unname(pmf_list[[1]]), sim_delay_pmf, tol = 0.02)
  expect_equal(unname(pmf_list[[2]]), sim_delay_pmf, tol = 0.02)
})

test_that("estimate_and_apply_delays uses correct rows", {
  # repeat with smaller number
  sim_delay_pmf <- c(0.25, 0.25, 0.25, 0.25)

  # Generate counts for each reference date
  counts <- 10 * c(4, 4, 4, 4, 4, 4, 4)

  # Create a complete triangle based on the known delay PMF
  complete_triangle <- lapply(counts, function(x) x * sim_delay_pmf)
  complete_triangle <- make_test_triangle(
    data = do.call(rbind, complete_triangle)
  )

  check_pmf <- estimate_delay(complete_triangle, n = 7)
  check_pmf

  # Create a reporting triangle with NAs in the lower right
  triangle <- apply_reporting_structure(complete_triangle)
  triangle

  slight_dif_triangle <- estimate_and_apply_delay(triangle)

  expect_equal(unclass(slight_dif_triangle), unclass(complete_triangle),
    tol = 0.5, check.attributes = FALSE
  )

  # Change entry in last row so that when used it wont estimate same delay
  triangle[1, 3] <- 3 * triangle[1, 3]
  triangle


  truncated_rts <- truncate_to_rows(triangle, n = 2)
  truncated_rts[1:2] # check that these all have 1s
  # These will always have the first row at the top. First one will be with
  # last row cut off, second will be with last 2 rows cut off

  retro_rts <- apply_reporting_structures(truncated_rts)
  retro_rts[1:2]
  # These look the same but with NAs in bottom right

  retro_pt_nowcast_mat_list <- estimate_and_apply_delays(
    retro_reporting_triangles = retro_rts,
    n = 5
  )
  # Get the empirical pmfs in your two pt nowcast matrices with the first row
  # included
  pmf_list <- lapply(retro_pt_nowcast_mat_list, estimate_delay)
  # Here we get back what we put in bc it doesn't use the first row
  expect_equal(unname(pmf_list[[1]]), sim_delay_pmf, tol = 0.06)
  # Here we don't bc we use the first row with the larger value
  expect_estimates_differ(unname(pmf_list[[2]]), sim_delay_pmf, tol = 0.06)
})
