% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/causalDisco-search.R
\name{CausalDiscoSearch}
\alias{CausalDiscoSearch}
\title{R6 Interface to causalDisco Search Algorithms}
\description{
This class implements the search algorithms from the \pkg{causalDisco}
package, which wraps and adds temporal order to \pkg{pcalg} algorithms.
It allows to set the data, sufficient statistics, test, score, and algorithm.
}
\examples{
# Generally, we do not recommend using the R6 classes directly, but rather
# use the disco() or any method function, for example pc(), instead.

data(tpc_example)

# background knowledge (tiers + one exogenous var)
kn <- knowledge(
  tpc_example,
  tier(
    child ~ starts_with("child"),
    youth ~ starts_with("youth"),
    old ~ starts_with("oldage")
  )
)

# Recommended (TPC example):
my_tpc <- tpc(engine = "causalDisco", test = "fisher_z", alpha = 0.05)
result <- disco(data = tpc_example, method = my_tpc, knowledge = kn)
plot(result)

# or
my_tpc <- my_tpc |>
  set_knowledge(kn)
result <- my_tpc(tpc_example)
plot(result)

# Using R6 class:

# --- Constraint-based: TPC ----------------------------------------------------
s_tpc <- CausalDiscoSearch$new()
s_tpc$set_params(list(verbose = FALSE))
s_tpc$set_test("fisher_z", alpha = 0.2)
s_tpc$set_alg("tpc")
s_tpc$set_knowledge(kn, directed_as_undirected = TRUE)
s_tpc$set_data(tpc_example)
res_tpc <- s_tpc$run_search()
print(res_tpc)

# Switch to TFCI on the same object (reuses suffStat/test)
s_tpc$set_alg("tfci")
res_tfci <- s_tpc$run_search()
print(res_tfci)

# --- Score-based: TGES --------------------------------------------------------
s_tges <- CausalDiscoSearch$new()
s_tges$set_score("tbic") # Gaussian temporal score
s_tges$set_alg("tges")
s_tges$set_data(tpc_example, set_suff_stat = FALSE) # suff stat not used for TGES
s_tges$set_knowledge(kn)
res_tges <- s_tges$run_search()
print(res_tges)

# --- Intentional error demonstrations ----------------------------------------

# run_search() without setting an algorithm
try(CausalDiscoSearch$new()$run_search(tpc_example))

# set_suff_stat() requires data and test first
s_err <- CausalDiscoSearch$new()
try(s_err$set_suff_stat()) # no data & no test
s_err$set_data(tpc_example, set_suff_stat = FALSE)
try(s_err$set_suff_stat()) # no test

# unknown test / score / algorithm
try(CausalDiscoSearch$new()$set_test("not_a_test"))
try(CausalDiscoSearch$new()$set_score("not_a_score"))
try(CausalDiscoSearch$new()$set_alg("not_an_alg"))

# set_knowledge() requires a `knowledge` object
try(CausalDiscoSearch$new()$set_knowledge(list(not = "Knowledge")))
}
\seealso{
\code{\link[=knowledge]{knowledge()}}.
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{data}}{A \code{data.frame} holding the data set currently attached to the
search object. Can be set with \code{set_data()}.}

\item{\code{score}}{A function that will be used to build the score,
when data is set. Can be set with \code{$set_score()}. Recognized values
are:
\itemize{
\item \code{"tbic"} - Temporal BIC score for Gaussian data.
See \link{TemporalBIC}.
\item \code{"tbdeu"} - Temporal BDeu score for discrete data.
See \link{TemporalBDeu}.
}}

\item{\code{test}}{A function that will be used to test independence.
Can be set with \code{$set_test()}. Recognized values are:
\itemize{
\item \code{"reg"} - Regression test for discrete or binary data.
See \code{\link[=reg_test]{reg_test()}}.
\item \code{"fisher_z"} - Fisher Z test for Gaussian data.
See \code{\link[=cor_test]{cor_test()}}.
}}

\item{\code{alg}}{A function that will be used to run the search algorithm.
Can be set with \code{$set_alg()}. Recognized values are:
\itemize{
\item \code{"tfci"} - TFCI algorithm.
See \code{\link[=tfci]{tfci()}}.
\item \code{"tges"} - TGES algorithm.
See \code{\link[=tges]{tges()}}.
\item \code{"tpc"}  - TPC algorithm.
See \code{\link[=tpc]{tpc()}}.
}}

\item{\code{params}}{A list of parameters for the test and algorithm.
Can be set with \code{$set_params()}.
TODO: not secure yet in terms of distributing arguments.
Use with caution.}

\item{\code{suff_stat}}{Sufficient statistic. The format and contents of the
sufficient statistic depends on which test is being used.}

\item{\code{knowledge}}{A \code{Knowledge} object holding background knowledge.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-CausalDiscoSearch-new}{\code{CausalDiscoSearch$new()}}
\item \href{#method-CausalDiscoSearch-set_params}{\code{CausalDiscoSearch$set_params()}}
\item \href{#method-CausalDiscoSearch-set_data}{\code{CausalDiscoSearch$set_data()}}
\item \href{#method-CausalDiscoSearch-set_suff_stat}{\code{CausalDiscoSearch$set_suff_stat()}}
\item \href{#method-CausalDiscoSearch-set_test}{\code{CausalDiscoSearch$set_test()}}
\item \href{#method-CausalDiscoSearch-set_score}{\code{CausalDiscoSearch$set_score()}}
\item \href{#method-CausalDiscoSearch-set_alg}{\code{CausalDiscoSearch$set_alg()}}
\item \href{#method-CausalDiscoSearch-set_knowledge}{\code{CausalDiscoSearch$set_knowledge()}}
\item \href{#method-CausalDiscoSearch-run_search}{\code{CausalDiscoSearch$run_search()}}
\item \href{#method-CausalDiscoSearch-clone}{\code{CausalDiscoSearch$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-new"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-new}{}}}
\subsection{Method \code{new()}}{
Constructor for the \code{CausalDiscoSearch} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-set_params"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-set_params}{}}}
\subsection{Method \code{set_params()}}{
Sets the parameters for the test and algorithm.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$set_params(params)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{params}}{A list of parameters to set.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-set_data"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-set_data}{}}}
\subsection{Method \code{set_data()}}{
Sets the data for the search algorithm.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$set_data(data, set_suff_stat = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{A \code{data.frame} or a \code{matrix} containing the data.}

\item{\code{set_suff_stat}}{Logical; whether to set the sufficient statistic.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-set_suff_stat"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-set_suff_stat}{}}}
\subsection{Method \code{set_suff_stat()}}{
Sets the sufficient statistic for the data.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$set_suff_stat()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-set_test"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-set_test}{}}}
\subsection{Method \code{set_test()}}{
Sets the test for the search algorithm.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$set_test(method, alpha = 0.05)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{A string specifying the type of test to use.}

\item{\code{alpha}}{Significance level for the test.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-set_score"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-set_score}{}}}
\subsection{Method \code{set_score()}}{
Sets the score for the search algorithm.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$set_score(method, params = list())}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{A string specifying the type of score to use.}

\item{\code{params}}{A list of parameters to pass to the score function.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-set_alg"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-set_alg}{}}}
\subsection{Method \code{set_alg()}}{
Sets the algorithm for the search.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$set_alg(method)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{A string specifying the type of algorithm to use.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-set_knowledge"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-set_knowledge}{}}}
\subsection{Method \code{set_knowledge()}}{
Sets the background knowledge for the search with a \code{Knowledge} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$set_knowledge(kn, directed_as_undirected = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{kn}}{A \code{Knowledge} object.}

\item{\code{directed_as_undirected}}{Logical; whether to treat directed edges in
the knowledge as undirected. Default is \code{FALSE}. This is due to the
nature of how \pkg{pcalg} handles background knowledge when using
\code{\link[pcalg:skeleton]{pcalg::skeleton()}} under the hood in
\code{\link[=tpc]{tpc()}} and
\code{\link[=tfci]{tfci()}}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-run_search"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-run_search}{}}}
\subsection{Method \code{run_search()}}{
Runs the search algorithm on the data.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$run_search(data = NULL, set_suff_stat = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{A \code{data.frame} or a \code{matrix} containing the data.}

\item{\code{set_suff_stat}}{Logical; whether to set the sufficient statistic}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-CausalDiscoSearch-clone"></a>}}
\if{latex}{\out{\hypertarget{method-CausalDiscoSearch-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{CausalDiscoSearch$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
