% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rem_stat_persistence.R
\name{remstats_persistence}
\alias{remstats_persistence}
\title{Compute Butts' (2008) Persistence Network Statistic for Event Dyads in a Relational Event Sequence}
\usage{
remstats_persistence(
  time,
  sender,
  target,
  sampled,
  observed,
  ref_sender = TRUE,
  nopastEvents = NA,
  dependency = FALSE,
  relationalTimeSpan = 0
)
}
\arguments{
\item{time}{The vector of event times from the post-processing event sequence.}

\item{sender}{The vector of event senders from the post-processing event sequence.}

\item{target}{The vector of event targets from the post-processing event sequence}

\item{sampled}{A vector for the post-processing event sequence where i is equal to 1 if the observed dyadic event is sampled and 0 if not.}

\item{observed}{A vector for the post-processing event sequence where i is equal to 1 if the dyadic event is observed and 0 if not.}

\item{ref_sender}{TRUE/FALSE. TRUE indicates that the persistence statistic will be computed in reference to the sender’s past relational history (see details section). FALSE indicates that the persistence statistic will be computed in reference to the target’s past relational history (see details section). Set to TRUE by default.}

\item{nopastEvents}{The numerical value that specifies what value should be given to events in which the sender has sent not past ties (i's neighborhood when sender = TRUE) or has not received any past ties (j's neighborhood when sender = FALSE). Set to NA by default.}

\item{dependency}{TRUE/FALSE. TRUE indicates that temporal relevancy will be modeled (see the details section). FALSE indicates that temporal relevancy will not be modeled, that is, all past events are relevant (see the details section). Set to FALSE by default.}

\item{relationalTimeSpan}{If dependency = TRUE, a numerical value that corresponds to the temporal span for relational relevancy, which must be the same measurement unit as the observed_time and processed_time objects. When dependency = TRUE, the relevant events are events that have occurred between current event time, \emph{t}, and \emph{t-relationalTimeSpan}. For example, if the time measurement is the number of days since the first event and the value for relationalTimeSpan is set to 10, then only those events which occurred in the past 10 days are included in the computation of the statistic.}
}
\value{
The vector of persistence network statistics for the relational event sequence.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

This function computes the persistence network sufficient statistic for
a relational event sequence (see Butts 2008). Persistence measures the proportion of past ties sent from the event sender that went to the current event receiver.
Furthermore, this measure allows for persistence scores to be only
computed for the sampled events, while creating the weights based on the full event
sequence. Moreover, the function allows users to specify relational relevancy for the resulting statistic.
}
\details{
The function calculates the persistence network sufficient statistic for a relational event sequence based on Butts (2008).

The formula for persistence for event \eqn{e_i} with reference to the sender's past relational history is:
\deqn{Persistence_{e_{i}} = \frac{d(s(e_{i}),r(e_{i}), A_t)}{d(s(e_{i}), A_t)} }

where  \eqn{d(s(e_{i}),r(e_{i}), A_t)} is the number of past events where the current event sender sent a tie to the current event receiver, and \eqn{d(s(e_{i}), A_t)} is the number of past events where the current sender sent a tie.

The formula for persistence for event \eqn{e_i} with reference to the target's past relational history is:
\deqn{Persistence_{e_{i}} = \frac{d(s(e_{i}),r(e_{i}), A_t)}{d(r(e_{i}), A_t)} }

where  \eqn{d(s(e_{i}),r(e_{i}), A_t)} is the number of past events where the current event sender sent a tie to the current event receiver, and \eqn{d(r(e_{i}), A_t)} is the number of past events where the current receiver recieved a tie.

Moreover, researchers interested in modeling temporal relevancy (see Quintane, Mood, Dunn, and Falzone 2022) can specify the relational time span, that is, length of time for which events are considered
relationally relevant. This should be specified via the option \emph{relationalTimeSpan} with \emph{dependency} set to TRUE.
}
\examples{


# A Dummy One-Mode Event Dataset
events <- data.frame(time = 1:18,
                                eventID = 1:18,
                                sender = c("A", "B", "C",
                                           "A", "D", "E",
                                           "F", "B", "A",
                                           "F", "D", "B",
                                           "G", "B", "D",
                                           "H", "A", "D"),
                                target = c("B", "C", "D",
                                           "E", "A", "F",
                                           "D", "A", "C",
                                           "G", "B", "C",
                                           "H", "J", "A",
                                           "F", "C", "B"))

# Creating the Post-Processing Event Dataset with Null Events
eventSet <- create_riskset(type = "one-mode",
                          time = events$time,
                          eventID = events$eventID,
                          sender = events$sender,
                          receiver = events$target,
                          p_samplingobserved = 1.00,
                          n_controls = 6,
                          seed = 9999)

#Computing the persistence statistic for the relational event sequence
eventSet$remstats_persistence <- remstats_persistence(
   time = as.numeric(eventSet$time),
   observed = eventSet$observed,
   sampled = rep(1,nrow(eventSet)),
   sender = eventSet$sender,
   target = eventSet$receiver,
   ref_sender = TRUE)

}
\references{
Butts, Carter T. 2008. "A relational event framework for social action." \emph{Sociological Methodology} 38(1): 155-200.

Quintane, Eric, Martin Wood, John Dunn, and Lucia Falzon. 2022. “Temporal
Brokering: A Measure of Brokerage as a Behavioral Process.” \emph{Organizational Research Methods}
25(3): 459-489.
}
\author{
Kevin A. Carson \href{mailto:kacarson@arizona.edu}{kacarson@arizona.edu}, Diego F. Leal \href{mailto:dflc@arizona.edu}{dflc@arizona.edu}
}
