#' Download data sets by id (source: I.Stat)
#'
#' @description
#' Download data sets from I.Stat (old ISTAT provider). Alternatively, use get_istatdata to download data sets from IstatData (new ISTAT provider).
#' The parameters "id_dataset" to download data sets can be found using list_i_stat function or search_i_stat function.
#'
#' @usage get_i_stat(id_dataset,
#'                      start_period = NULL,
#'                      end_period = NULL,
#'                      recent = FALSE,
#'                      csv = FALSE,
#'                      xlsx = FALSE,
#'                      lang = "both",
#'                      cache = FALSE,
#'                      update_cache = FALSE,
#'                      compress_file = FALSE,
#'                      cache_dir = "cache_dir")
#' @param id_dataset Data set id.
#' @param start_period Time value for the start (NULL by default).
#' @param end_period Time value for the end (NULL bu default).
#' @param recent FALSE by default, if TRUE, the function retrieves data from last 10 years.
#' @param csv FALSE by default, if TRUE, the function saves the data set to directory as .csv.
#' @param xlsx FALSE by default, if TRUE, the function saves the data set to directory as .xlsx.
#' @param lang Language parameter for labels ("ita" for Italian, "eng" for English). By default both languages are returned. 
#' @param cache FALSE by default. If TRUE, downloaded data are cached in the directory specified by `cache_dir`.
#' @param update_cache FALSE by default. If TRUE, the cache is updated.
#' @param compress_file FALSE by default. It compresses the RDS file when caching. Only used when `cache = TRUE`.
#' @param cache_dir By default it saves the cache directory into the current working directory when `cache = TRUE`. The directory is created if it does not exist.
#'
#' @return It returns the data set as data.frame. It can be saved to environment or as .csv/.xlsx.
#' @importFrom rsdmx readSDMX
#' @importFrom utils write.csv
#' @importFrom writexl write_xlsx
#' @export
#'
#' @note
#' Downloading may take some time. Future versions will speed up the process.
#'
#' @examples \dontrun{get_i_stat("12_60", 
#'                               start_period = 2015, 
#'                               end_period = 2018, 
#'                               lang = "eng", 
#'                               cache = FALSE)}
get_i_stat <- function(id_dataset,
                       start_period = NULL,
                       end_period = NULL,
                       recent = FALSE,
                       csv = FALSE,
                       xlsx = FALSE,
                       lang = "both",
                       cache = FALSE,
                       update_cache = FALSE,
                       compress_file = FALSE,
                       cache_dir = "cache_dir") {

  if (cache && !dir.exists(cache_dir)) {
    dir.create(cache_dir)
  }

  query <- list(
    id_dataset = id_dataset,
    start_period = start_period,
    end_period = end_period,
    recent = recent,
    lang = lang
  )

  query_hash <- digest::digest(query, algo = "md5")

  cache_file <- file.path(cache_dir, paste0(query_hash, ".rds"))

  if (cache && file.exists(cache_file) && !update_cache) {
    message("Reading from cache: ", cache_file)
    dati <- readRDS(cache_file)
    return(dati)
  }

  if (!is.null(start_period) & !is.null(end_period) & isTRUE(recent)) {
    stop("Use recent = TRUE for data from last 10 years.")
  }
  
  if (!curl::has_internet()) {
    message(
      "No Internet connection available."
    )
    return(invisible(NULL))
  }
  
  if (is.null(start_period) & is.null(end_period)) {
    result <- try({

      dati <- readSDMX(providerId = "ISTAT_LEGACY", resource = "data",
                       flowRef = id_dataset, dsd = TRUE)
      dati <- as.data.frame(dati, labels = TRUE)
    }, silent = FALSE)

  } else if (!is.null(start_period) & !is.null(end_period)) {
    result <- try({
      dati <- readSDMX(providerId = "ISTAT_LEGACY", resource = "data",
                       flowRef = id_dataset, dsd = TRUE, start = start_period, end = end_period)
      dati <- as.data.frame(dati, labels = TRUE)
    }, silent = FALSE)

  } else if (is.null(start_period) & is.null(end_period) & isTRUE(recent)) {
    result <- try({

      dati <- readSDMX(providerId = "ISTAT_LEGACY", resource = "data",
                       flowRef = id_dataset, dsd = TRUE,
                       start = as.numeric(format(Sys.Date(), "%Y")) - 10,
                       end = as.numeric(format(Sys.Date(), "%Y")))
      dati <- as.data.frame(dati, labels = TRUE)
    }, silent = FALSE)
  }
  
  if (inherits(result, "try-error")) {
    message(
      "Unable to retrieve dataset '", id_dataset, "' from dati.istat.it. ",
      "The dataset may not exist, may have changed, or the service may be temporarily unavailable."
    )
    return(invisible(NULL))
  }
  
  if (lang == "ita") {
    dati <- dati[, !grepl("\\.en$", names(dati))]
  } else if (lang == "eng") {
    dati <- dati[, !grepl("\\.it$", names(dati))]
  }

  if (cache) {
    if (update_cache || !file.exists(cache_file)) {
      saveRDS(dati, file = cache_file, compress = compress_file)
      message("Data cached at: ", cache_file)
    }
  }


  if (isTRUE(csv) & isTRUE(xlsx)) {
    stop("Choose csv OR xlsx")
  } else if (isTRUE(xlsx)) {
    write_xlsx(dati, "Dataset.xlsx")
    return(dati)
  } else if (isTRUE(csv)) {
    write.csv(dati, "Dataset.csv")
    return(dati)
  } else {
    return(dati)
  }
}

