#' Load Bundled Example Dataset
#'
#' @description
#' Convenience function to load the bundled CKD/CVD/Diabetes multi-state
#' cohort data (2,000 patients, 3 biomarkers, 9 transitions) from the
#' CSV files shipped with the package. Returns both the longitudinal
#' biomarker data and survival event data in a list, ready for use
#' with \code{\link{jmSurf}}.
#'
#' The longitudinal data contains 68,112 biomarker measurements (eGFR,
#' BNP, HbA1c) and the survival data contains 4,701 rows covering 9
#' transition types across CKD, CVD, Diabetes, and At-risk states.
#'
#' @return A list with two data frames:
#'   \item{long_data}{Longitudinal biomarker measurements (68,112 rows, 5 columns:
#'     \code{patient_id}, \code{visit_time_years}, \code{biomarker},
#'     \code{value}, \code{unit})}
#'   \item{surv_data}{Survival/transition events (4,701 rows, 17 columns including
#'     \code{patient_id}, \code{start_time}, \code{stop_time}, \code{status},
#'     \code{state_from}, \code{state_to}, \code{transition},
#'     \code{age_baseline}, \code{sex}, \code{bmi})}
#'
#' @examples
#' \donttest{
#' dat <- load_example_data()
#' str(dat$long_data)
#' str(dat$surv_data)
#'
#' # Fit a model directly
#' fit <- jmSurf(dat$long_data, dat$surv_data, verbose = FALSE)
#' summary(fit)
#' }
#'
#' @export
load_example_data <- function() {

  ## Read from bundled CSV files
  long_path <- system.file("extdata", "longitudinal_biomarkers.csv",
                           package = "jmSurface")
  surv_path <- system.file("extdata", "survival_events.csv",
                           package = "jmSurface")
  if (long_path == "" || surv_path == "")
    stop("CSV files not found. Reinstall jmSurface.")

  long_data <- utils::read.csv(long_path, stringsAsFactors = FALSE)
  surv_data <- utils::read.csv(surv_path, stringsAsFactors = FALSE)

  ## Normalize arrow encoding
  surv_data$transition <- gsub("\u2192", "->", surv_data$transition)

  message("Loaded: ", length(unique(long_data$patient_id)), " patients, ",
          nrow(long_data), " biomarker rows, ",
          nrow(surv_data), " survival rows")

  list(long_data = long_data, surv_data = surv_data)
}
