% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dig.R
\name{dig}
\alias{dig}
\title{Search for patterns of a custom type}
\usage{
dig(
  x,
  f,
  condition = everything(),
  focus = NULL,
  disjoint = var_names(colnames(x)),
  excluded = NULL,
  min_length = 0,
  max_length = Inf,
  min_support = 0,
  min_focus_support = 0,
  min_conditional_focus_support = 0,
  max_support = 1,
  filter_empty_foci = FALSE,
  t_norm = "goguen",
  max_results = Inf,
  verbose = FALSE,
  threads = 1L,
  error_context = list(arg_x = "x", arg_f = "f", arg_condition = "condition", arg_focus =
    "focus", arg_disjoint = "disjoint", arg_excluded = "excluded", arg_min_length =
    "min_length", arg_max_length = "max_length", arg_min_support = "min_support",
    arg_min_focus_support = "min_focus_support", arg_min_conditional_focus_support =
    "min_conditional_focus_support", arg_max_support = "max_support",
    arg_filter_empty_foci = "filter_empty_foci", arg_t_norm = "t_norm", arg_max_results =
    "max_results", arg_verbose = "verbose", 
     arg_threads = "threads", call =
    current_env())
)
}
\arguments{
\item{x}{A matrix or data frame. If a matrix, it must be numeric (double) or
logical. If a data frame, all columns must be numeric (double) or logical.}

\item{f}{A callback function executed for each generated condition. It may
declare any subset of the arguments listed below. The algorithm detects
which arguments are present and provides only those values to \code{f}. This
design allows the user to control both the amount of information received
and the computational cost, as some arguments are more expensive to
compute than others. The function \code{f} is expected to return an object
(typically a list) representing a pattern or patterns related to the
condition. The results of all calls of \code{f} are collected and returned as
a list. Possible arguments are: \code{condition}, \code{sum}, \code{support}, \code{indices},
\code{weights}, \code{pp}, \code{pn}, \code{np}, \code{nn}, or \code{foci_supports} (deprecated), which
are thoroughly described below in the "Details" section.}

\item{condition}{tidyselect expression (see
\href{https://tidyselect.r-lib.org/articles/syntax.html}{tidyselect syntax})
specifying columns of \code{x} to use as condition predicates}

\item{focus}{tidyselect expression (see
\href{https://tidyselect.r-lib.org/articles/syntax.html}{tidyselect syntax})
specifying columns of \code{x} to use as focus predicates}

\item{disjoint}{An atomic vector (length = number of columns in \code{x}) defining
groups of predicates. Columns in the same group cannot appear together in
a condition. With data from \code{\link[=partition]{partition()}}, use \code{\link[=var_names]{var_names()}} on column
names to construct \code{disjoint}.}

\item{excluded}{\code{NULL} or a list of character vectors, each representing an
implication formula. In each vector, all but the last element form the
antecedent and the last element is the consequent. These formulae are
treated as \emph{tautologies} and used to filter out generated conditions. If
a condition contains both the antecedent and the consequent of any such
formula, it is not passed to the callback function \code{f}. Likewise, if a
condition contains the antecedent, the corresponding focus (the consequent)
is not passed to \code{f}.}

\item{min_length}{Minimum number of predicates in a condition required to
trigger the callback \code{f}. Must be \eqn{\ge 0}. If set to 0, the empty
condition also triggers the callback.}

\item{max_length}{Maximum number of predicates allowed in a condition.
Conditions longer than \code{max_length} are not generated. If \code{Inf}, the only
limit is the total number of available predicates. Must be \eqn{\ge 0} and
\eqn{\ge min_length}. This setting strongly influences both the number of
generated conditions and the speed of the search.}

\item{min_support}{Minimum support of a condition required to trigger \code{f}.
Support is the relative frequency of the condition in \code{x}. For logical
data, this is the proportion of rows where all condition predicates are
\code{TRUE}. For numeric (double) data, support is the mean (over all rows) of
the products of predicate values. Must be in \eqn{[0,1]}. If a condition’s
support falls below \code{min_support}, recursive generation of its extensions
is stopped. Thus, \code{min_support} directly affects search speed and the
number of callback calls.}

\item{min_focus_support}{Minimum support of a focus required for it to be
passed to \code{f}. For logical data, this is the proportion of rows where both
the condition and the focus are \code{TRUE}. For numeric (double) data, support
is computed as the mean (over all rows) of a t-norm of predicate values
(the t-norm is selected by \code{t_norm}). Must be in \eqn{[0,1]}. Foci with
support below this threshold are excluded. Together with
\code{filter_empty_foci}, this parameter influences both search speed and the
number of triggered calls of \code{f}.}

\item{min_conditional_focus_support}{Minimum conditional support of a focus
within a condition. Defined as the relative frequency of rows where the
focus is \code{TRUE} among those where the condition is \code{TRUE}. If \eqn{sum}
(see \code{support} in \emph{Details}) is the number of rows (or sum of truth
degrees for fuzzy data) satisfying the condition, and \eqn{pp} (see
\code{pp[i]} in \emph{Details}) is the sum of truth degrees where both the condition
and the focus hold, then conditional support is \eqn{pp / sum}. Must be in
\eqn{[0,1]}. Foci below this threshold are not passed to \code{f}. Together with
\code{filter_empty_foci}, this parameter influences search speed and the number
of callback calls.}

\item{max_support}{Maximum support of a condition to trigger \code{f}. Conditions
with support above this threshold are skipped, but recursive generation of
their supersets continues. Must be in \eqn{[0,1]}.}

\item{filter_empty_foci}{Logical; controls whether \code{f} is triggered for
conditions with no remaining foci after filtering by \code{min_focus_support}
or \code{min_conditional_focus_support}. If \code{TRUE}, \code{f} is called only when at
least one focus remains. If \code{FALSE}, \code{f} is called regardless.}

\item{t_norm}{T-norm used for conjunction of weights: \code{"goedel"} (minimum),
\code{"goguen"} (product), or \code{"lukas"} (Łukasiewicz).}

\item{max_results}{Maximum number of results (objects returned by the
callback \code{f}) to store and return in the output list. When this limit
is reached, generation of further conditions stops. Use a positive
integer to enable early stopping; set to \code{Inf} to remove the cap.}

\item{verbose}{Logical; if \code{TRUE}, print progress messages.}

\item{threads}{Number of threads for parallel computation.}

\item{error_context}{A list of details to be used when constructing error
messages. This is mainly useful when \code{dig()} is called from another
function and errors should refer to the caller’s argument names rather
than those of \code{dig()}. The list must contain:
\itemize{
\item \code{arg_x} – name of the argument \code{x} as a character string
\item \code{arg_f} – name of the argument \code{f} as a character string
\item \code{arg_condition} – name of the argument \code{condition}
\item \code{arg_focus} – name of the argument \code{focus}
\item \code{arg_disjoint} – name of the argument \code{disjoint}
\item \code{arg_excluded} – name of the argument \code{excluded}
\item \code{arg_min_length} – name of the argument \code{min_length}
\item \code{arg_max_length} – name of the argument \code{max_length}
\item \code{arg_min_support} – name of the argument \code{min_support}
\item \code{arg_min_focus_support} – name of the argument
\code{min_focus_support}
\item \code{arg_min_conditional_focus_support} – name of the argument
\code{min_conditional_focus_support}
\item \code{arg_max_support} – name of the argument \code{max_support}
\item \code{arg_filter_empty_foci} – name of the argument \code{filter_empty_foci}
\item \code{arg_t_norm} – name of the argument \code{t_norm}
\item \code{arg_threads} – name of the argument \code{threads}
\item \code{call} – environment in which to evaluate error messages
}}
}
\value{
A list of results returned by the callback function \code{f}.
}
\description{
A general function for searching for patterns of a custom type. The function
allows selection of columns of \code{x} to be used as condition predicates. It
enumerates all possible conditions in the form of elementary conjunctions of
selected predicates, and for each condition executes a user-defined callback
function \code{f}. The callback is expected to perform some analysis and return an
object (often a list) representing a pattern or patterns related to the
condition. The results of all calls are returned as a list.

The callback function \code{f} may accept a number of arguments (see \code{f} argument
description). The algorithm automatically provides condition-related
information to \code{f} based on which arguments are present.

In addition to conditions, the function can evaluate \emph{focus} predicates
(foci). Foci are specified separately and are tested within each generated
condition. Extra information about them is then passed to \code{f}.

Restrictions may be imposed on generated conditions, such as:
\itemize{
\item minimum and maximum condition length (\code{min_length}, \code{max_length});
\item minimum condition support (\code{min_support});
\item minimum focus support (\code{min_focus_support}), i.e. support of rows where
both the condition and the focus hold.
}
}
\details{
Let \eqn{P} be the set of condition predicates selected by \code{condition} and
\eqn{E} be the set of focus predicates selected by \code{focus}. The function
generates all possible conditions as elementary conjunctions of distinct
predicates from \eqn{P}. These conditions are filtered using \code{disjoint},
\code{excluded}, \code{min_length}, \code{max_length}, \code{min_support}, and \code{max_support}.

For each remaining condition, all foci from \eqn{E} are tested and filtered
using \code{min_focus_support} and \code{min_conditional_focus_support}. If at least
one focus remains (or if \code{filter_empty_foci = FALSE}), the callback \code{f} is
executed with details of the condition and foci. Results of all calls are
collected and returned as a list.

Let \eqn{C} be a condition (\eqn{C \subseteq P}), \eqn{F} the set of
filtered foci (\eqn{F \subseteq E}), \eqn{R} the set of rows of \code{x}, and
\eqn{\mu_C(r)} the truth degree of condition \eqn{C} on row \eqn{r}. The
parameters passed to \code{f} are defined as:
\itemize{
\item \code{condition}: a named integer vector of column indices representing the
predicates of \eqn{C}. Names correspond to column names.
\item \code{sum}: a numeric scalar value of the number of rows satisfying \eqn{C} for
logical data, or the sum of truth degrees for fuzzy data,
\eqn{sum = \sum_{r \in R} \mu_C(r)}.
\item \code{support}: a numeric scalar value of relative frequency of rows satisfying \eqn{C},
\eqn{supp = sum / |R|}.
\item \code{pp}, \code{pn}, \code{np}, \code{nn}: a numeric vector of entries of a contingency table
for \eqn{C} and \eqn{F}, satisfying the Ruspini condition
\eqn{pp + pn + np + nn = |R|}.
The \eqn{i}-th elements of these vectors correspond to the \eqn{i}-th focus
\eqn{F_i} from \eqn{F} and are defined as:
\itemize{
\item \code{pp[i]}: rows satisfying both \eqn{C} and \eqn{F_i},
\eqn{pp_i = \sum_{r \in R} \mu_{C \land F_i}(r)}.
\item \code{pn[i]}: rows satisfying \eqn{C} but not \eqn{F_i},
\eqn{pn_i = \sum_{r \in R} \mu_C(r) - pp_i}.
\item \code{np[i]}: rows satisfying \eqn{F_i} but not \eqn{C},
\eqn{np_i = \sum_{r \in R} \mu_{F_i}(r) - pp_i}.
\item \code{nn[i]}: rows satisfying neither \eqn{C} nor \eqn{F_i},
\eqn{nn_i = |R| - (pp_i + pn_i + np_i)}.
}
}
}
\examples{
library(tibble)

# Prepare iris data
d <- partition(iris, .breaks = 2)

# Simple callback: return formatted condition names
dig(x = d,
    f = function(condition) format_condition(names(condition)),
    min_support = 0.5)

# Callback returning condition and support
res <- dig(x = d,
           f = function(condition, support) {
               list(condition = format_condition(names(condition)),
                    support = support)
           },
           min_support = 0.5)
do.call(rbind, lapply(res, as_tibble))

# Within each condition, evaluate also supports of columns starting with
# "Species"
res <- dig(x = d,
           f = function(condition, support, pp) {
               c(list(condition = format_condition(names(condition))),
                 list(condition_support = support),
                 as.list(pp / nrow(d)))
           },
           condition = !starts_with("Species"),
           focus = starts_with("Species"),
           min_support = 0.5,
           min_focus_support = 0)
do.call(rbind, lapply(res, as_tibble))

# Multiple patterns per condition based on foci
res <- dig(x = d,
           f = function(condition, support, pp) {
               lapply(seq_along(pp), function(i) {
                   list(condition = format_condition(names(condition)),
                        condition_support = support,
                        focus = names(pp)[i],
                        focus_support = pp[[i]] / nrow(d))
               })
           },
           condition = !starts_with("Species"),
           focus = starts_with("Species"),
           min_support = 0.5,
           min_focus_support = 0)

# Flatten result and convert to tibble
res <- unlist(res, recursive = FALSE)
do.call(rbind, lapply(res, as_tibble))

}
\seealso{
\code{\link[=partition]{partition()}}, \code{\link[=var_names]{var_names()}}, \code{\link[=dig_grid]{dig_grid()}}
}
\author{
Michal Burda
}
