% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simpls.fit.R
\name{simpls.fit}
\alias{simpls.fit}
\title{Sijmen de Jong's SIMPLS}
\usage{
simpls.fit(
  X,
  Y,
  ncomp,
  center = TRUE,
  orthScores = FALSE,
  stripped = FALSE,
  ...
)
}
\arguments{
\item{X}{a matrix of observations.  \code{NA}s and \code{Inf}s are not
allowed.}

\item{Y}{a vector or matrix of responses.  \code{NA}s and \code{Inf}s are
not allowed.}

\item{ncomp}{the number of components to be used in the modelling.}

\item{center}{logical, determines if the \eqn{X} and \eqn{Y} matrices are
mean centered or not. Default is to perform mean centering.}

\item{orthScores}{logical.  If \code{TRUE} the scores are orthogonalised for
increased numerical precision (default = FALSE, for speed).}

\item{stripped}{logical.  If \code{TRUE} the calculations are stripped as
much as possible for speed; this is meant for use with cross-validation or
simulations when only the coefficients are needed.  Defaults to
\code{FALSE}.}

\item{\dots}{other arguments.  Currently ignored.}
}
\value{
A list containing the following components is returned:
\item{coefficients}{an array of regression coefficients for 1, \ldots{},
\code{ncomp} components.  The dimensions of \code{coefficients} are
\code{c(nvar, npred, ncomp)} with \code{nvar} the number of \code{X}
variables and \code{npred} the number of variables to be predicted in
\code{Y}.} \item{scores}{a matrix of scores.} \item{loadings}{a matrix of
loadings.} \item{Yscores}{a matrix of Y-scores.} \item{Yloadings}{a matrix
of Y-loadings.} \item{projection}{the projection matrix used to convert X to
scores.} \item{Xmeans}{a vector of means of the X variables.}
\item{Ymeans}{a vector of means of the Y variables.} \item{fitted.values}{an
array of fitted values.  The dimensions of \code{fitted.values} are
\code{c(nobj, npred, ncomp)} with \code{nobj} the number samples and
\code{npred} the number of Y variables.} \item{residuals}{an array of
regression residuals.  It has the same dimensions as \code{fitted.values}.}
\item{Xvar}{a vector with the amount of X-variance explained by each
component.} \item{Xtotvar}{Total variance in \code{X}.}

If \code{stripped} is \code{TRUE}, only the components \code{coefficients},
\code{Xmeans} and \code{Ymeans} are returned.
}
\description{
Fits a PLSR model with the SIMPLS algorithm.
}
\details{
This function should not be called directly, but through the generic
functions \code{plsr} or \code{mvr} with the argument
\code{method="simpls"}.  SIMPLS is much faster than the NIPALS algorithm,
especially when the number of X variables increases, but gives slightly
different results in the case of multivariate Y.  SIMPLS truly maximises the
covariance criterion.  According to de Jong, the standard PLS2 algorithms
lie closer to ordinary least-squares regression where a precise fit is
sought; SIMPLS lies closer to PCR with stable predictions.
}
\examples{
## Simulation of SIMPLS stability
# The graphics produced, demonstrate the numeric instability of the original
# SIMPLS without score orthogonalization.
set.seed(42)
N <- 100
p <- 2000
ncomp <- 40
simData <- data.frame(X = I(matrix(rnorm(N*p),N)), y = rnorm(N))
pls    <- plsr(y ~ X, data=simData, ncomp=ncomp)
simps  <- plsr(y ~ X, data=simData, ncomp=ncomp, method="simpls")
simpsO <- plsr(y ~ X, data=simData, ncomp=ncomp, method="simpls", orthScores=TRUE)
normScores <- pls$scores
for(i in 1:ncomp)
  normScores[,i] <- normScores[,i]/sqrt(sum(normScores[,i]^2))

# Check number of equal digits
eqDig <- function(x,y){
  xy <- abs(x - y)
  xy[xy == 0] <- 10^-16
  -colMeans(log10(xy))
}
eqDig_PLS_oSIMPLS    <- eqDig(normScores, simpsO$scores)
eqDig_SIMPLS_PLS     <- eqDig(simps$scores, normScores)
eqDig_SIMPLS_oSIMPLS <- eqDig(simps$scores, simpsO$scores)
# Correlation between models
cor_PLS_oSIMPLS    <- diag(cor(pls$scores,simps$scores))
cor_SIMPLS_oSIMPLS <- diag(cor(pls$scores,simpsO$scores))
cor_SIMPLS_PLS     <- diag(cor(simps$scores,simpsO$scores))

par.old <- par(mfrow=c(2,1), mar=c(4,4,1,1), las=1)
matplot(2:ncomp,cbind(eqDig_PLS_oSIMPLS,eqDig_SIMPLS_PLS, eqDig_SIMPLS_oSIMPLS)[-1,],
        type="l", xlab="component", ylab="equal digits", ylim=c(0,17),
        panel.first=grid())
legend("bottomleft", legend=c("PLS, SIMPLS", "PLS, OrthSIMPLS", "SIMPLS, OrthSIMPLS"),
       col=1:3, lty=1:3)
matplot(1:ncomp,cbind(cor_PLS_oSIMPLS,cor_SIMPLS_oSIMPLS,cor_SIMPLS_PLS), type="l",
        ylab="correlation", xlab="component", panel.first=grid())
legend("bottomleft", legend=c("PLS, SIMPLS", "PLS, OrthSIMPLS", "SIMPLS, OrthSIMPLS"),
       col=1:3, lty=1:3)
par(par.old)
}
\references{
de Jong, S. (1993) SIMPLS: an alternative approach to partial
least squares regression.  \emph{Chemometrics and Intelligent Laboratory
Systems}, \bold{18}, 251--263.
}
\seealso{
\code{\link{mvr}} \code{\link{plsr}} \code{\link{pcr}}
\code{\link{kernelpls.fit}} \code{\link{widekernelpls.fit}}
\code{\link{oscorespls.fit}}
}
\author{
Ron Wehrens and Bjørn-Helge Mevik
}
\keyword{multivariate}
\keyword{regression}
