# https://data.worldpop.org/GIS/AgeSex_structures/Global_2015_2030/R2025A/2020/DEU/v1/1km_ua/constrained/deu_f_00_2020_CN_1km_R2025A_UA_v1.tif
# https://data.worldpop.org/GIS/AgeSex_structures/Global_2015_2030/R2025A/2020/DEU/v1/1km_ua/constrained/deu_f_05_2020_CN_1km_R2025A_UA_v1.tif

# {iso}_{gender}_{age group}_{year}_{type}_{resolution}_{release}_{version}.tif

# age_group 00,01,05,10,...,90
# gender = f,m

# year = 2015 -> 2030

# https://data.worldpop.org/GIS/Population/Global_2015_2030/R2025A/2020/DEU/v1/1km_ua/constrained/deu_pop_2020_CN_1km_R2025A_UA_v1.tif

# https://www-genesis.destatis.de/

#https://ergebnisse.zensus2022.de/datenbank/online/statistic/1000A/table/1000A-1008/table-toolbar#filter=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&sortByValue=default

#' Infer and fit a population model from `SurvStat` output
#'
#' `SurvStat` can be queried for count or incidence. From the combination of
#' these metrics queried across the whole range of disease notifications for any
#' given year we can infer a stratified population size, that `SurvStat` is using
#' to calculate it's incidence. This is simply modelled with a local polynomial
#' over time to allow us to fill in weekly population denominators.
#'
#' @param count_df a dataframe from the output of `get_timeseries()` or `get_snapshot()`
#' @inheritParams get_timeseries
#'
#' @returns the `count_df` dataframe with an additional `population` column
#' @export
#' @concept survstat
#'
#' @examples
#' \donttest{
#'
#' # snapshot:
#' get_snapshot(
#'   disease = diseases$`COVID-19`,
#'   geography = "state",
#'   season=2024
#' ) %>%
#' fit_population() %>%
#' dplyr::glimpse()
#'
#' # timeseries
#' # A weekly population estimate is inferred from the yearly data:
#' get_timeseries(
#'   diseases$`COVID-19`,
#'   measure = "Count",
#'   age_group = age_groups$children_coarse
#' ) %>%
#' fit_population() %>%
#' dplyr::glimpse()
#'
#' }
fit_population = function(count_df, .progress = TRUE) {
  age_group = if ("age_code" %in% colnames(count_df)) {
    unique(stats::na.omit(stringr::str_extract(
      count_df$age_code,
      "^(.*)\\.&\\[.+\\]$",
      1
    )))
  } else {
    NULL
  }

  geography = if ("geo_code" %in% colnames(count_df)) {
    unique(stats::na.omit(stringr::str_extract(
      count_df$geo_code,
      "^(.*)\\.&\\[.+\\]$",
      1
    )))
  } else {
    NULL
  }

  if ("year" %in% colnames(count_df)) {
    years = unique(count_df$year)
  } else {
    years = unique(as.numeric(format(count_df$date, "%Y")))
  }
  pop_df = infer_population(
    age_group = age_group,
    geography = geography,
    years = years,
    .progress = .progress
  )
  by = intersect(colnames(pop_df), colnames(count_df))

  if (!"date" %in% colnames(count_df)) {
    return(
      count_df %>%
        dplyr::inner_join(pop_df, by = by)
    )
  }

  if (length(years) == 1) {
    return(
      count_df %>%
        dplyr::inner_join(pop_df %>% dplyr::select(-year), by = by)
    )
  }

  model_df = pop_df %>%
    tidyr::nest(data = c(population, year)) %>%
    dplyr::mutate(
      model = purrr::map(
        data,
        ~ suppressWarnings(locfit::locfit(population ~ locfit::lp(year), .x))
      )
    )

  out = count_df %>%
    dplyr::mutate(
      year = as.numeric(date - as.Date("2020-07-01")) / 365.25 + 2020
    ) %>%
    tidyr::nest(
      new_data = dplyr::any_of(c("year", "date", "count", "incidence"))
    ) %>%
    dplyr::inner_join(model_df %>% dplyr::select(-data), by = by) %>%
    dplyr::mutate(
      data2 = purrr::map2(
        model,
        new_data,
        ~ {
          .y %>% dplyr::mutate(population = stats::predict(.x, newdata = .y))
        }
      )
    ) %>%
    dplyr::select(-model, -new_data) %>%
    tidyr::unnest(data2) %>%
    dplyr::select(-year)

  return(out)
}


#' @describeIn fit_population Query `SurvStat` for data to impute a population denominator
#'
#' @inheritParams get_timeseries
#' @param geography (optional) one of `"state"`, `"nuts"`, or `"county"` to define the
#'   resolution of the query. Does not accept a `sf` map or subset of
#'   (unlike `get_timeseries()`).
#'
#' @returns a dataframe with geography, age grouping, year and population columns
#' @export
#'
#' @examples
#' \donttest{
#' infer_population(years=2020:2025) %>% dplyr::glimpse()
#' }
infer_population = function(
  age_group = NULL,
  geography = NULL,
  years = NULL,
  .progress = TRUE
) {
  #
  if (!is.null(geography)) {
    if (geography %in% names(geography_resolution)) {
      geography = geography_resolution[[geography]]
    }
    colhier = geography
  } else {
    colhier = NULL
  }

  if (!is.null(age_group)) {
    if (age_group %in% names(age_groups)) {
      rowhier = age_groups[[age_group]]
    } else {
      rowhier = age_group
    }
  } else {
    rowhier = NULL
  }

  this_year = as.numeric(format(Sys.Date(), "%Y"))
  if (is.null(years)) {
    years = 2001:this_year
  }
  years = .year_filter(years)
  out = NULL

  if (.progress) {
    cli::cli_progress_bar(total = length(years))
  }

  # Change the cache settings:
  old_cache_settings = set_cache_settings(stale = Inf)
  on.exit(set_cache_settings(old_cache_settings), add = TRUE)

  for (year in years) {
    if (year$values$year < this_year) {
      # Never clear cache of old years
      set_cache_settings(stale = Inf)
    } else {
      set_cache_settings(stale = 14)
    }

    count_req = .get_request(
      commands$olap_data,
      cube = cubes$survstat,
      language = languages$german,
      column_hierarchy = colhier,
      measure = "Count",
      filters = year,
      row_hierarchy = rowhier
    )

    # cat(as.character(count_req))

    count_res = count_req %>%
      .do_survstat_command(quiet = TRUE)
    count_df = count_res %>%
      .process_olap_data_result() %>%
      dplyr::rename(count = value)

    incid_req = .get_request(
      commands$olap_data,
      cube = cubes$survstat,
      language = languages$german,
      column_hierarchy = colhier,
      measure = "Incidence",
      filters = year,
      row_hierarchy = rowhier
    )
    # cat(as.character(incid_req))

    incid_res = incid_req %>% .do_survstat_command(quiet = TRUE)
    incid_df = incid_res %>%
      .process_olap_data_result() %>%
      dplyr::rename(incid = value)

    if (.progress) {
      cli::cli_progress_update()
    }

    join_cols = c(
      if (!is.null(colhier)) c("col_code", "col_name") else character(),
      if (!is.null(rowhier)) c("row_code", "row_name") else character()
    )
    if (length(join_cols) > 0) {
      pop_df = count_df %>%
        dplyr::inner_join(incid_df, by = join_cols) %>%
        dplyr::select(dplyr::all_of(c(join_cols, "count", "incid")))
    } else {
      pop_df = count_df %>% dplyr::cross_join(incid_df)
    }

    pop_df = pop_df %>%
      dplyr::mutate(
        population = count / incid * 100000,
        !!!year$values
      ) %>%
      dplyr::filter(!is.na(population)) %>%
      dplyr::select(-count, -incid)

    out = dplyr::bind_rows(pop_df, out)
  }

  if ("col_name" %in% colnames(out)) {
    out = out %>% dplyr::rename(geo_name = col_name, geo_code = col_code)
  }

  if ("row_name" %in% colnames(out)) {
    out = out %>% dplyr::rename(age_name = row_name, age_code = row_code)
  }

  if ("age_code" %in% colnames(out)) {
    tmp = .fmt_range(out$age_code)
    out = out %>% dplyr::mutate(!!!tmp)
  }

  out = out %>%
    dplyr::group_by(dplyr::across(-dplyr::any_of(c("population", "year"))))

  # Interpolate to dates...?

  if (.progress) {
    cli::cli_progress_done()
  }

  return(out)
}
