\name{info}
\alias{info}
\alias{print.info}
\title{Retrieve Information About Sumerian Signs}
\description{
Analyzes a transliterated Sumerian text string and retrieves detailed information about each sign, including syllabic readings, sign names, cuneiform symbols, and alternative readings.

The function \code{info} computes the result and returns an object of class \code{"info"}. The \code{print} method displays a summary of different text representations in the console.
}
\usage{
info(x, mapping = NULL)

\method{print}{info}(x, flatten = FALSE, ...)
}
\arguments{
\item{x}{
    For \code{info}: a character string of length 1 containing transliterated Sumerian text.

    For \code{print.info}: an object of class \code{"info"}.
  }
  \item{mapping}{A data frame containing the sign mapping table with columns \code{syllables}, \code{name}, and \code{cuneiform}. If \code{NULL} (the default), the package's internal mapping file \file{etcsl_mapping.txt} is loaded.}
  \item{flatten}{Logical. If \code{TRUE}, grammar indicators in the text are removed (such as parentheses, brackets, braces, and operators). If \code{FALSE} (the default), the original separators are preserved.}
  \item{...}{Additional arguments passed to the print method (currently unused).}
}
\details{
The function \code{info} performs the following steps:

\enumerate{
  \item Splits the input string into signs and separators using \code{\link{split_sumerian}}.
  \item Standardizes the signs.
  \item Looks up each sign in the mapping table based on its type:
    \itemize{
      \item Type 1 (lowercase): Searches for a matching syllable reading.
      \item Type 2 (uppercase): Searches for a matching sign name.
      \item Type 3 (cuneiform): Searches for a matching cuneiform character.
    }
  \item Returns a data frame with the results, along with the separators stored as an attribute.
}

The mapping table must contain the following columns:
\describe{
  \item{syllables}{Comma-separated list of possible syllabic readings for the sign. The first reading is used as the default.}
  \item{name}{The canonical sign name in uppercase.}
  \item{cuneiform}{The Unicode cuneiform character.}
}

The \code{print} method displays each sign with its name and alternative readings, followed by three text representations: syllables, sign names, and cuneiform text.
}
\value{
\code{info} returns a data frame of class \code{c("info", "data.frame")} with one row per sign and the following columns:
  \item{reading}{The syllabic reading of the sign. For lowercase input, this is the standardized input; for other types, this is the default syllable from the mapping.}
  \item{sign}{The Unicode cuneiform character corresponding to the sign.}
  \item{name}{The canonical sign name in uppercase.}
  \item{alternatives}{A comma-separated string of all possible syllabic readings for the sign.}

The data frame has an attribute \code{"separators"} containing the separator characters between signs.

\code{print.info} prints the following to the console and returns \code{x} invisibly:
\describe{
  \item{Sign table}{Each sign with its cuneiform symbol, name, and alternative readings.}
  \item{syllables}{The text with syllabic readings, using hyphens as separators within words.}
  \item{sign names}{The text with sign names, using periods as separators within words.}
  \item{cuneiform text}{The text rendered in Unicode cuneiform characters, with hyphens and periods removed.}
}
}
\note{
If no custom mapping is provided, the function loads the internal mapping file included with the \pkg{sumer} package.
}
\seealso{
\code{\link{split_sumerian}} for splitting Sumerian text into signs,
}
\examples{
library(stringr)

# Basic usage - compute and print
info("lugal-e")

# Store the result for further processing
result <- info("an-ki")
result

# Access the underlying data frame
result$sign
result$name

# Print with and without flattened separators
result <- info("(an)na")
print(result)
print(result, flatten = TRUE)

# Using a custom mapping table
path <- system.file("extdata", "etcsl_mapping.txt", package = "sumer")
my_mapping <- read.csv2(path, sep=";", na.strings="")
info("an-ki", mapping = my_mapping)
}
\keyword{utilities}
\keyword{character}
