#' @importFrom dplyr rename
#' @importFrom stats is.leaf hclust as.dendrogram as.dist
#' 
#' @title get_tsi_helper
#' 
#' @author Johannes De Groeve
#' @description get the time since isolation on a tabs object or recvect object 
#'
#' @param filename path where files or qs2/rds was exported, reconstruct object of class tabs or recvect object 
#' @param verbose boolean, print messages
#' @param extant boolean, only for currently existing shapes (TRUE) or all shapes (FALSE)
#' @param recname recname, default NULL will create a tree for every merged polygon
#'
#' @return list of dendrogram objects
#' 
#' @noRd
#' @keywords internal
#' 
get_tsi_helper <- function(filename,
                           extant=TRUE,
                           recname=NULL,
                           verbose=FALSE){
  
  if(is.character(filename)){ # if character, calculate area from an already exported object 
    reconstruction <- import(filename)
    recvect <- reconstruction$recvect
  } else {
    if(inherits(filename,'SpatVectorCollection')){
      recvect <- filename
    } else {
      # if reconstruct
      recvect <- filename$recvect  
    }
  }
  
  recvect <- vect(recvect)
  
  groups <- get_groups(filename,unnest=FALSE)
  ids <- get_groups(filename,unnest=TRUE)
  
  if(is.null(recname)){
    recvect_groups <- regroup(filename, 
                              by='group',
                              group=TRUE,
                              extant=extant)
  } else{
    recvect_groups <- regroup(filename, 
                              by='recname',
                              recname=recname,
                              group=TRUE,
                              extant=extant)  
  }
  
  dend_l <- list()
  vv <- c()
  for(y in 1:length(recvect_groups)){
    # list all islands
    print(y)
    islands <- unique(recvect_groups[[y]]$recid)
    names(islands) <- unique(recvect_groups[[y]]$recname)
    
    # if(length(islands) <= 1){  # open - check number of islands 
    #   message(paste0('single isolated island ',y))
    #   vv <- c(vv,FALSE)
    #   
    #   #dend_l[[i]] <- NA
    # } else { # open - tree generated if there is more than 1 island
      vv <- c(vv,TRUE)
      # split off from Ibiza
      time_since_isolation_l <- list()
      # compute for all island combinations the connection history
      for ( i in 1:length(islands)){
        bool <- grepl(islands[i], recvect$recnames)
        
        time_since_isolation <- sapply(islands, function(x){
          bool2 <- grepl(x, recvect$recnames) # which recnames contain x? 
          
          sub <- recvect[which(bool & bool2),]
          periods <- sub$period
          period <- sort(-as.numeric(gsub('[A-Z]','',periods)),decreasing = TRUE)[1] # periods
          return(period)
        })
        time_since_isolation_l[[i]] <- time_since_isolation
      }
      names(time_since_isolation_l) <- names(islands)
      
      #time_since_isolation_l
      
      # Convert to a full matrix
      mat <- matrix(NA, nrow = length(islands), ncol = length(islands),
                    dimnames = list(names(islands), 
                                    names(islands)))
      
      for (i in names(islands)) {
        mat[i, names(time_since_isolation_l[[i]])] <- time_since_isolation_l[[i]]
      }
      
      # Fill symmetric lower triangle from upper triangle
      for (i in 1:nrow(mat)) {
        for (j in 1:ncol(mat)) {
          if (is.na(mat[i,j]) && !is.na(mat[j,i])) {
            mat[i,j] <- mat[j,i]
          }
        }
      }
      
      # Ensure symmetry
      mat[upper.tri(mat)] <- t(mat)[upper.tri(mat)]
      
      # Convert times (negative BP) to positive distances
      dist_mat <- abs(mat)
      
      # Convert to 'dist' object
      d <- stats::as.dist(dist_mat)
      # Hierarchical clustering
      #hc <- hclust(d, method = "single")  # 'single' linkage matches your connection-first logic
    
      dend_l[[y]] <- d
    #} 
    }# close - run per group
    
      #dend_l <- dend_l[ !(sapply(dend_l, is.null) | sapply(dend_l, length) == 0) ]
      names(dend_l) <- names(recvect_groups)[vv]
    return(dend_l)
  }

#' @title get_tsi
#' 
#' @author Johannes De Groeve
#' @description get the time since isolation on a tabs object or recvect object 
#'
#' @param filename path where files or qs2/rds was exported, reconstruct object of class tabs or recvect object 
#' @param verbose boolean, print messages
#' @param extant boolean, only for currently existing shapes (TRUE) or all shapes (FALSE)
#' @param recname recname, default NULL will create a tree for every merged polygon
#'
#' @return list of dendrogram objects
#' 
#' @export
#' 
#' @examples
#' \dontrun{
#' # load data samples
#' sporades <- sporades()
#' topo <- sporades$topo
#' labs <- sporades$labs
#' curve <- sporades$curve
#' 
#' 
#' # change duplicated names 
#' duplicates <- labs[labs$name == 'UNKNOWN',]$name
#' labs[labs$name == 'UNKNOWN',]$name <- paste0(duplicates,' ',toupper(letters[1:length(duplicates)]))
#' 
#' # reconstruct
#' rec <- reconstruct(topo=topo,
#'                    region=labs,
#'                    curve=curve,
#'                    reclabs='name'
#'                    )
#' 
#' # present day shapes only
#' tsi_e <- get_tsi(rec,
#'                  extant=TRUE
#'                  )
#'                       
#' # present and past shapes               
#' tsi_a <- get_tsi(rec,
#'                  extant=FALSE
#'                  )
#'          
#' # select specific shape for which the connectivity change is calculated
#' tsi <- get_tsi(rec,
#'                recname=rec$recvect$AP0000000$recname[6],
#'                extant=FALSE)
#' }
get_tsi <- function(filename,
                    extant=TRUE,
                    recname=NULL,
                    verbose=FALSE){
  
  dist_l <- get_tsi_helper(filename=filename,
                           extant=extant,
                           recname=recname,
                           verbose=verbose)
  dist_l <- dist_l[ !(sapply(dist_l, is.null) | sapply(dist_l, length) == 0) ]
  return(dist_l)
}


#' @title get_tree
#' 
#' @author Johannes De Groeve
#' @description get the tree based on a tabs object or recvect object 
#'
#' @param filename path where files or qs2/rds was exported, reconstruct object of class tabs or recvect object 
#' @param verbose boolean, print messages
#' @param extant boolean, only for currently existing shapes (TRUE) or all shapes (FALSE)
#' @param ultrametric boolean, if TRUE, all the tips (leaves) are the same distance from the root
#' @param recname recname, default NULL will create a tree for every merged polygon
#'
#' @return list of dendrogram objects
#' 
#' @export
#' 
#' @examples
#' \dontrun{
#' # load data samples
#' sporades <- sporades()
#' topo <- sporades$topo
#' labs <- sporades$labs
#' curve <- sporades$curve
#' 
#' 
#' # change duplicated names 
#' duplicates <- labs[labs$name == 'UNKNOWN',]$name
#' labs[labs$name == 'UNKNOWN',]$name <- paste0(duplicates,' ',toupper(letters[1:length(duplicates)]))
#' 
#' # reconstruct
#' rec <- reconstruct(topo=topo,
#'                    region=labs,
#'                    curve=curve,
#'                    reclabs='name'
#'                    )
#' 
#' # present day shapes only
#' trees_e <- get_tree(rec,
#'                     extant=TRUE,
#'                     ultrametric=FALSE)
#' trees_e_u <- get_tree(rec,
#'                       extant=TRUE,
#'                       ultrametric=TRUE)
#'                       
#' par(mar=c(4,4,2,8),mfrow=c(1,2), oma = c(0, 0, 4, 0))
#' plot(trees_e[[2]], horiz = TRUE, 
#'      main = "Observed tree", xlab = "Years BP")
#' plot(trees_e_u[[2]], horiz = TRUE, 
#'      main = "Ultrametric tree", xlab = "Years BP")                  
#' mtext("Connectivity change over time", outer = TRUE, cex = 1.5, line = 1) 
#' mtext("present-day islands", outer = TRUE, cex = 1, line = -0.2) 
#'             
#' # present and past shapes               
#' trees_a <- get_tree(rec,
#'                     extant=FALSE,
#'                     ultrametric=FALSE)
#' trees_a_u <- get_tree(rec,
#'                       extant=FALSE,
#'                       ultrametric=TRUE)
#'          
#' par(mar=c(4,0,2,8),mfrow=c(1,2), oma = c(0, 0, 4, 0))
#' plot(trees_a[[2]], 
#'      horiz = TRUE, 
#'      main = "Observed tree", 
#'      xlab = "Years BP")
#' plot(trees_a_u[[2]], 
#'      horiz = TRUE, 
#'      main = "Ultrametric tree", 
#'      xlab = "Years BP")
#' mtext("Connectivity change over time", outer = TRUE, cex = 1.5, line = 1) 
#' mtext("present and past islands", outer = TRUE, cex = 1, line = -0.2) 
#'                      
#' # select specific shape for which the connectivity change is plotted
#' tree <- get_tree(rec,
#'                  recname=rec$recvect$AP0000000$recname[6],
#'                  extant=FALSE)
#' par(mar=c(4,0,2,8),mfrow=c(1,1), oma = c(0, 0, 0, 0))
#' plot(tree$`Nisí Peristéra`, 
#'      horiz = TRUE, 
#'      main = "Observed tree", 
#'      xlab = "Years BP")
#' }
get_tree <- function(filename,
                     extant=TRUE,
                     recname=NULL,
                     ultrametric=FALSE,
                     verbose=FALSE){
  
  if(is.character(filename)){ # if character, calculate area from an already exported object 
    reconstruction <- import(filename)
    recvect <- reconstruction$recvect
  } else {
    if(inherits(filename,'SpatVectorCollection')){
      recvect <- filename
    } else {
      # if reconstruct
      recvect <- filename$recvect  
    }
  }
  
  recvect <- vect(recvect)
  
  groups <- get_groups(filename,unnest=FALSE)
  ids <- get_groups(filename,unnest=TRUE)
  
  if(is.null(recname)){
    recvect_groups <- regroup(filename, 
                              by='group',
                              group=TRUE,
                              extant=extant)
  } else{
    recvect_groups <- regroup(filename, 
                              by='recname',
                              recname=recname,
                              group=TRUE,
                              extant=extant)  
  }
  
  # obtain the distance matrix (time since isolation) 
  # tsi_helper keeps all the island clusters, also the ones with a single island
  dist_l <- get_tsi_helper(filename=filename,
                           extant=extant,
                           recname=recname,
                           verbose=verbose)
  
  # turn distance matrices into trees 
  dend_l <- list()
  vv <- c()
  for(y in 1:length(dist_l)){
    # list all islands
    islands <- unique(recvect_groups[[y]]$recid)
    names(islands) <- unique(recvect_groups[[y]]$recname)
    
    if(length(islands) <= 1){  # open - check number of islands 
      message(paste0('single isolated island ',y))
      vv <- c(vv,FALSE)
      
      #dend_l[[i]] <- NA
    } else { # open - tree generated if there is more than 1 island
      vv <- c(vv,TRUE)
      
      # Hierarchical clustering
      hc <- stats::hclust(dist_l[[y]], method = "single")  # 'single' linkage matches your connection-first logic
      
      # # Plot dendrogram with reversed time axis (BP)
      # par(mar=c(4,2,2,8))
      
      if (ultrametric) { # open - if extant 
        dend <- as.dendrogram(hc)
        dend_l[[y]] <- dend
        # Horizontal, reversed axis
        # plot(dend, horiz = TRUE, xlim = rev(range(hc$height)), 
        #      main = "Connectivity Dendrogram", xlab = "Years BP")
      } else { # else - if past islands are included, adjust dendrogram 
        
        # Example: get drowning time per island (max period where island still exists)
        drowning_times <- sapply(islands, function(isl){
          bool <- grepl(isl, recvect$recnames)
          if (!any(bool)) return(NA)
          # get the max period (closest to 0) when island was connected (or still exists)
          min_period <- min(as.numeric(gsub('[A-Z]', '', recvect$period[bool])))
          return(min_period)
        })
        
        # Before calling, store original heights from hc dendrogram
        dend <- as.dendrogram(hc)
        # Copy original heights into dend nodes so we can reference them
        dend <- store_original_heights(dend)
        # Then call adjust_leaf_heights and refer to height_orig to avoid branch shrinkage:
        dend3 <- adjust_leaf_heights_v2(dend, drowning_times)
        dend_l[[y]] <- dend3
        
        # plot(dend3, horiz = TRUE, xlim = rev(range(hc$height)), 
        #      main = "Connectivity Dendrogram (Leaves Adjusted with Original Branch Lengths)", xlab = "Years BP")
      } # close - if past islands are included, adjust dendrogram 
    } # close - tree generated
  }
  
  dend_l <- dend_l[ !(sapply(dend_l, is.null) | sapply(dend_l, length) == 0) ]
  names(dend_l) <- names(recvect_groups)[vv]
  
  return(dend_l)
}






#' @title store_original_heights
#' 
#' @author Johannes De Groeve
#' @description helper function to store original heights of a dendrogram
#'
#' @param node 
#'
#' @return dendrogram object 
#' 
#' @noRd
#' @keywords internal
#'
store_original_heights <- function(node) {
  if (!stats::is.leaf(node)) {
    node[[1]] <- store_original_heights(node[[1]])
    node[[2]] <- store_original_heights(node[[2]])
  }
  attr(node, "height_orig") <- attr(node, "height")
  return(node)
}

#' @title adjust_leaf_heights_v2
#' 
#' @author Johannes De Groeve
#' @description helper function to adjust the leaf heights for non-extant shapes.
#'
#' @param node 
#' @param drowning_times
#'
#' @return dendrogram object 
#' 
#' @noRd
#' @keywords internal
#'
adjust_leaf_heights_v2 <- function(node, drowning_times) {
  if (stats::is.leaf(node)) {
    lbl <- attr(node, "label")
    drown_time <- drowning_times[lbl]
    attr(node, "height") <- ifelse(!is.na(drown_time) && drown_time > 0, drown_time, 0)
  } else {
    node[[1]] <- adjust_leaf_heights_v2(node[[1]], drowning_times)
    node[[2]] <- adjust_leaf_heights_v2(node[[2]], drowning_times)
    
    h1 <- attr(node[[1]], "height")
    h2 <- attr(node[[2]], "height")
    children_max <- max(h1, h2)
    original_height <- attr(node, "height_orig")
    attr(node, "height") <- max(children_max, original_height)
  }
  return(node)
}


#' #' @title get_tree_groups
#' #' 
#' #' @author Johannes De Groeve
#' #' @description get the tree based on a tabs object or recvect object 
#' #'
#' #' @param filename path where files or qs2/rds was exported, reconstruct object of class tabs or recvect object 
#' #' @param verbose boolean, print messages
#' #' @param extant boolean, only for currently existing shapes (TRUE) or all shapes (FALSE)
#' #' @param ultrametric boolean, if TRUE, all the tips (leaves) are the same distance from the root
#' #' @param recname recname, default NULL will create a tree for every merged polygon
#' #'
#' #' @return list of dendrogram objects
#' #' 
#' #' @export
#' #' 
#' #' @examples
#' #' # load data samples
#' #' sporades <- sporades()
#' #' topo <- sporades$topo
#' #' labs <- sporades$labs
#' #' curve <- sporades$curve
#' #' 
#' #' 
#' #' # change duplicated names 
#' #' duplicates <- labs[labs$name == 'UNKNOWN',]$name
#' #' labs[labs$name == 'UNKNOWN',]$name <- paste0(duplicates,' ',toupper(letters[1:length(duplicates)]))
#' #' 
#' #' # reconstruct
#' #' rec <- reconstruct(topo=topo,
#' #'                    region=labs,
#' #'                    curve=curve,
#' #'                    reclabs='name'
#' #'                    )
#' #' 
#' #' # present day shapes only
#' #' trees_e <- get_tree(rec,
#' #'                     extant=TRUE,
#' #'                     ultrametric=FALSE)
#' #' trees_e_u <- get_tree(rec,
#' #'                       extant=TRUE,
#' #'                       ultrametric=TRUE)
#' #'                       
#' #' par(mar=c(4,4,2,8),mfrow=c(1,2), oma = c(0, 0, 4, 0))
#' #' plot(trees_e[[2]], horiz = TRUE, 
#' #'      main = "Observed tree", xlab = "Years BP")
#' #' plot(trees_e_u[[2]], horiz = TRUE, 
#' #'      main = "Ultrametric tree", xlab = "Years BP")                  
#' #' mtext("Connectivity change over time", outer = TRUE, cex = 1.5, line = 1) 
#' #' mtext("present-day islands", outer = TRUE, cex = 1, line = -0.2) 
#' #' tsi_groups_e <- get_tree_groups(trees_e[[2]])
#' #' tsi_groups_e_u <- get_tree_groups(trees_e_u[[2]])
#' #' 
#' #' # present and past shapes               
#' #' trees_a <- get_tree(rec,
#' #'                     extant=FALSE,
#' #'                     ultrametric=FALSE)
#' #' trees_a_u <- get_tree(rec,
#' #'                       extant=FALSE,
#' #'                       ultrametric=TRUE)
#' #'          
#' #' par(mar=c(4,0,2,8),mfrow=c(1,2), oma = c(0, 0, 4, 0))
#' #' plot(trees_a[[2]], 
#' #'      horiz = TRUE, 
#' #'      main = "Observed tree", 
#' #'      xlab = "Years BP")
#' #' plot(trees_a_u[[2]], 
#' #'      horiz = TRUE, 
#' #'      main = "Ultrametric tree", 
#' #'      xlab = "Years BP")
#' #' mtext("Connectivity change over time", outer = TRUE, cex = 1.5, line = 1) 
#' #' mtext("present and past islands", outer = TRUE, cex = 1, line = -0.2) 
#' #' tsi_groups_a <- get_tree_groups(trees_a[[2]])
#' #' tsi_groups_a_u <- get_tree_groups(trees_a_u[[2]])
#' #'                      
#' #' # select specific shape for which the connectivity change is plotted
#' #' tree <- get_tree(rec,
#' #'                  recname=rec$recvect$AP0000000$recname[6],
#' #'                  extant=FALSE)
#' #' par(mar=c(4,0,2,8),mfrow=c(1,1), oma = c(0, 0, 0, 0))
#' #' plot(tree$`Nisí Peristéra`, 
#' #'      horiz = TRUE, 
#' #'      main = "Observed tree", 
#' #'      xlab = "Years BP")
#' #' tsi_groups <- get_tree_groups(tree[[1]])
#' #'  
#' get_tree_groups <- function(tree, include_zero = TRUE, include_indices = FALSE) {
#'   
#'   # Coerce to hclust
#'   if (inherits(tree, "dendrogram")) {
#'     hc <- as.hclust(tree)
#'   } else if (inherits(tree, "hclust")) {
#'     hc <- tree
#'   } else stop("Provide a 'dendrogram' or 'hclust' object.")
#'   
#'   # The labels in the *correct* order are hc$labels
#'   labs <- hc$labels
#'   n <- length(labs)
#'   
#'   # Unique merge heights
#'   hs <- sort(unique(hc$height))
#'   if (include_zero) hs <- c(0, hs)
#'   
#'   res <- setNames(vector("list", length(hs)), as.character(hs))
#'   
#'   for (i in seq_along(hs)) {
#'     h <- hs[i]
#'     cl <- stats::cutree(hc, h = h)  # integer cluster id for each leaf (length n)
#'     
#'     # Split by position, so duplicate labels are handled correctly
#'     idx_by_cluster <- split(seq_len(n), cl)
#'     
#'     if (!include_indices) {
#'       res[[i]] <- lapply(idx_by_cluster, function(ix) labs[ix])
#'     } else {
#'       res[[i]] <- lapply(idx_by_cluster, function(ix)
#'         list(index = ix, label = labs[ix])
#'       )
#'     }
#'   }
#'   
#'   attr(res, "heights") <- hs
#'   return(res)
#' }
#' 
#' 
#' 


# #' @title get_tree
# #' 
# #' @author Johannes De Groeve
# #' @description get the tree based on a tabs object or recvect object
# #' 
# #' @param filename path where files or qs2/rds was exported, reconstruct object of class tabs or recvect object
# #' @param verbose boolean, print messages
# #' @param extant boolean, only for currently existing shapes (TRUE) or all shapes (FALSE)
# #' @param ultrametric boolean, if TRUE, all the tips (leaves) are the same distance from the root
# #' @param recname recname, default NULL will create a tree for every merged polygon
# #' 
# #' @return list of dendrogram objects
# 
# get_tree <- function(filename,
#                      extant=TRUE,
#                      recname=NULL,
#                      ultrametric=FALSE,
#                      verbose=FALSE){
#   
#   if(is.character(filename)){ # if character, calculate area from an already exported object 
#     reconstruction <- import(filename)
#     recvect <- reconstruction$recvect
#   } else {
#     if(inherits(filename,'SpatVectorCollection')){
#       recvect <- filename
#     } else {
#       # if reconstruct
#       recvect <- filename$recvect  
#     }
#   }
#   
#   recvect <- vect(recvect)
#   
#   groups <- get_groups(filename,unnest=FALSE)
#   ids <- get_groups(filename,unnest=TRUE)
#   
#   if(is.null(recname)){
#     recvect_groups <- regroup(filename, 
#                               by='group',
#                               group=TRUE,
#                               extant=extant)
#   } else{
#     recvect_groups <- regroup(filename, 
#                               by='recname',
#                               recname=recname,
#                               group=TRUE,
#                               extant=extant)  
#   }
#   
#   dend_l <- list()
#   vv <- c()
#   for(y in 1:length(recvect_groups)){
#     # list all islands
#     islands <- unique(recvect_groups[[y]]$recid)
#     names(islands) <- unique(recvect_groups[[y]]$recname)
#     
#     if(length(islands) <= 1){  # open - check number of islands 
#       message(paste0('single isolated island ',y))
#       vv <- c(vv,FALSE)
#       
#       #dend_l[[i]] <- NA
#     } else { # open - tree generated if there is more than 1 island
#       vv <- c(vv,TRUE)
#       # split off from Ibiza
#       time_since_isolation_l <- list()
#       # compute for all island combinations the connection history
#       for ( i in 1:length(islands)){
#         bool <- grepl(islands[i], recvect$recnames)
#         
#         time_since_isolation <- sapply(islands, function(x){
#           bool2 <- grepl(x, recvect$recnames) # which recnames contain x? 
#           
#           sub <- recvect[which(bool & bool2),]
#           periods <- sub$period
#           period <- sort(-as.numeric(gsub('[A-Z]','',periods)),decreasing = TRUE)[1] # periods
#           return(period)
#         })
#         time_since_isolation_l[[i]] <- time_since_isolation
#       }
#       names(time_since_isolation_l) <- names(islands)
#       
#       #time_since_isolation_l
#       
#       # Convert to a full matrix
#       mat <- matrix(NA, nrow = length(islands), ncol = length(islands),
#                     dimnames = list(names(islands), 
#                                     names(islands)))
#       
#       for (i in names(islands)) {
#         mat[i, names(time_since_isolation_l[[i]])] <- time_since_isolation_l[[i]]
#       }
#       
#       # Fill symmetric lower triangle from upper triangle
#       for (i in 1:nrow(mat)) {
#         for (j in 1:ncol(mat)) {
#           if (is.na(mat[i,j]) && !is.na(mat[j,i])) {
#             mat[i,j] <- mat[j,i]
#           }
#         }
#       }
#       
#       # Ensure symmetry
#       mat[upper.tri(mat)] <- t(mat)[upper.tri(mat)]
#       
#       # Convert times (negative BP) to positive distances
#       dist_mat <- abs(mat)
#       
#       # Convert to 'dist' object
#       d <- as.dist(dist_mat)
#       # Hierarchical clustering
#       hc <- hclust(d, method = "single")  # 'single' linkage matches your connection-first logic
#       
#       # # Plot dendrogram with reversed time axis (BP)
#       # par(mar=c(4,2,2,8))
#       
#       if (ultrametric) { # open - if extant 
#         dend <- as.dendrogram(hc)
#         dend_l[[y]] <- dend
#         # Horizontal, reversed axis
#         # plot(dend, horiz = TRUE, xlim = rev(range(hc$height)), 
#         #      main = "Connectivity Dendrogram", xlab = "Years BP")
#       } else { # else - if past islands are included, adjust dendrogram 
#         
#         # Example: get drowning time per island (max period where island still exists)
#         drowning_times <- sapply(islands, function(isl) {
#           bool <- grepl(isl, recvect$recnames)
#           if (!any(bool)) return(NA)
#           # get the max period (closest to 0) when island was connected (or still exists)
#           min_period <- min(as.numeric(gsub('[A-Z]', '', recvect$period[bool])))
#           return(min_period)
#         })
#         
#         # Before calling, store original heights from hc dendrogram
#         dend <- as.dendrogram(hc)
#         # Copy original heights into dend nodes so we can reference them
#         dend <- store_original_heights(dend)
#         # Then call adjust_leaf_heights and refer to height_orig to avoid branch shrinkage:
#         dend3 <- adjust_leaf_heights_v2(dend, drowning_times)
#         dend_l[[y]] <- dend3
#         
#         # plot(dend3, horiz = TRUE, xlim = rev(range(hc$height)), 
#         #      main = "Connectivity Dendrogram (Leaves Adjusted with Original Branch Lengths)", xlab = "Years BP")
#       } # close - if past islands are included, adjust dendrogram 
#     } # close - tree generated
#   }
#   
#   
#   dend_l <- dend_l[ !(sapply(dend_l, is.null) | sapply(dend_l, length) == 0) ]
#   names(dend_l) <- names(recvect_groups)[vv]
#   
#   return(dend_l)
# }
