use super::scalar::{Rint, Scalar};
use super::*;
use extendr_ffi::{
    dataptr, R_xlen_t, INTEGER_GET_REGION, INTEGER_IS_SORTED, INTEGER_NO_NA, SET_INTEGER_ELT,
};
use std::iter::FromIterator;














#[derive(PartialEq, Clone)]
pub struct Integers {
    pub(crate) robj: Robj,
}

use extendr_ffi::SEXPTYPE::INTSXP;
macros::gen_vector_wrapper_impl!(
    vector_type: Integers, 
    scalar_type: Rint,     
    primitive_type: i32,   
    r_prefix: INTEGER,     
    SEXP: INTSXP,          
    doc_name: integer,     
    altrep_constructor: make_altinteger_from_iterator,
);

macros::gen_from_iterator_impl!(
    vector_type: Integers,
    collect_from_type: i32,
    underlying_type: i32,
    SEXP: INTSXP,
    assignment: |dest: &mut i32, val: i32| *dest = val
);

impl Integers {
    
    pub fn get_region(&self, index: usize, dest: &mut [Rint]) -> usize {
        unsafe {
            let ptr: *mut i32 = dest.as_mut_ptr() as *mut i32;
            INTEGER_GET_REGION(self.get(), index as R_xlen_t, dest.len() as R_xlen_t, ptr) as usize
        }
    }

    
    pub fn is_sorted(&self) -> Rbool {
        unsafe { INTEGER_IS_SORTED(self.get()).into() }
    }

    
    pub fn no_na(&self) -> Rbool {
        unsafe { INTEGER_NO_NA(self.get()).into() }
    }
}


impl Integers {
    pub fn set_elt(&mut self, index: usize, val: Rint) {
        single_threaded(|| unsafe {
            SET_INTEGER_ELT(self.get(), index as R_xlen_t, val.inner());
        })
    }
}

impl Deref for Integers {
    type Target = [Rint];

    
    fn deref(&self) -> &Self::Target {
        unsafe {
            let ptr = dataptr(self.get()) as *const Rint;
            std::slice::from_raw_parts(ptr, self.len())
        }
    }
}

impl DerefMut for Integers {
    
    fn deref_mut(&mut self) -> &mut Self::Target {
        unsafe {
            let ptr = dataptr(self.get_mut()) as *mut Rint;
            std::slice::from_raw_parts_mut(ptr, self.len())
        }
    }
}

impl std::fmt::Debug for Integers {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        if self.len() == 1 {
            write!(f, "{:?}", self.elt(0))
        } else {
            f.debug_list().entries(self.iter()).finish()
        }
    }
}

impl TryFrom<Vec<i32>> for Integers {
    type Error = Error;

    fn try_from(value: Vec<i32>) -> std::result::Result<Self, Self::Error> {
        Ok(Self { robj: value.into() })
    }
}

#[cfg(test)]
mod tests {
    use crate as extendr_api;
    use crate::prelude::*;

    #[test]
    fn from_iterator() {
        test! {
            let vec : Integers = (0..3).collect();
            assert_eq!(vec, Integers::from_values([0, 1, 2]));
        }
    }

    #[test]
    fn iter_mut() {
        test! {
            let mut vec = Integers::from_values(0..3);
            vec.iter_mut().for_each(|v| *v += 1);
            assert_eq!(vec, Integers::from_values(1..4));
        }
    }

    #[test]
    fn iter() {
        test! {
            let vec = Integers::from_values(0..3);
            assert_eq!(vec.iter().sum::<Rint>(), 3);
        }
    }

    #[test]
    fn from_values_short() {
        test! {
            
            let vec = Integers::from_values((0..3).map(|i| 2-i));
            assert_eq!(vec.is_altrep(), false);
            assert_eq!(r!(vec.clone()), r!([2, 1, 0]));
            assert_eq!(vec.elt(1), 1);
            let mut dest = [0.into(); 2];
            vec.get_region(1, &mut dest);
            assert_eq!(dest, [1, 0]);
        }
    }

    #[test]
    fn from_values_altrep() {
        test! {
            let vec = Integers::from_values_altrep(0..1000000000);
            assert_eq!(vec.is_altrep(), true);
            assert_eq!(vec.elt(12345678), 12345678);
            let mut dest = [0.into(); 2];
            vec.get_region(12345678, &mut dest);
            assert_eq!(dest, [12345678, 12345679]);
        }
    }

    #[test]
    fn new() {
        test! {
            let vec = Integers::new(10);
            assert_eq!(vec.is_integer(), true);
            assert_eq!(vec.len(), 10);
        }
    }

    #[test]
    fn test_vec_i32_integers_conversion() {
        test! {
            let int_vec = vec![3,4,0,-2];
            let int_vec_robj: Robj = int_vec.clone().try_into().unwrap();
            
            assert_eq!(int_vec_robj.as_integer_slice().unwrap(), &int_vec);
        }
    }
}
