

/*
Run-time CPU feature detection on AArch64 Fuchsia by using zx_system_get_features.

As of nightly-2024-09-07, is_aarch64_feature_detected doesn't support run-time detection on Fuchsia.
https:

Refs:
- https:
- https:
*/

include!("common.rs");

#[allow(non_camel_case_types)]
mod ffi {
    
    pub(crate) type zx_status_t = i32;

    sys_const!({
        
        pub(crate) const ZX_OK: zx_status_t = 0;

        
        pub(crate) const ZX_FEATURE_KIND_CPU: u32 = 0;
        pub(crate) const ZX_ARM64_FEATURE_ISA_ATOMICS: u32 = 1 << 8;
    });

    
    #[link(name = "zircon")]
    extern "C" {
        
        pub(crate) fn zx_system_get_features(kind: u32, features: *mut u32) -> zx_status_t;
    }
}

fn zx_system_get_features(kind: u32) -> u32 {
    let mut out = 0_u32;
    
    let res = unsafe { ffi::zx_system_get_features(kind, &mut out) };
    if res != ffi::ZX_OK {
        return 0;
    }
    out
}

#[cold]
fn _detect(info: &mut CpuInfo) {
    let features = zx_system_get_features(ffi::ZX_FEATURE_KIND_CPU);
    if features & ffi::ZX_ARM64_FEATURE_ISA_ATOMICS != 0 {
        info.set(CpuInfo::HAS_LSE);
    }
}

#[allow(
    clippy::alloc_instead_of_core,
    clippy::std_instead_of_alloc,
    clippy::std_instead_of_core,
    clippy::undocumented_unsafe_blocks,
    clippy::wildcard_imports
)]
#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_fuchsia() {
        let features = zx_system_get_features(ffi::ZX_FEATURE_KIND_CPU);
        assert_ne!(features, 0);
        std::eprintln!("features: {:b}", features);
    }

    
    
    
    
    
    
    
    
    
    
    #[allow(clippy::cast_possible_wrap, clippy::cast_sign_loss, clippy::cast_possible_truncation)]
    const _: fn() = || {
        use test_helper::sys;
        
        let _: ffi::zx_status_t = 0 as sys::zx_status_t;
    };
}
